--
-- AdditionalSettingsManager
--
-- @author Rockstar
-- @date 27/03/2021
--


AdditionalSettingsManager = {
	LOAD_STATE = {
		NO_LOAD = 0,
		LOAD_MAP = 1,
		LOAD_MAP_FINISHED = 2,
		MISSION_START = 3
	}
}

local AdditionalSettingsManager_mt = Class(AdditionalSettingsManager)

function AdditionalSettingsManager:new(modDir, modName)
	local self = setmetatable({}, AdditionalSettingsManager_mt)

	AdditionalSettingsUtil.registerEvent("onLoad")
	AdditionalSettingsUtil.registerEvent("onLoadFinished")
	AdditionalSettingsUtil.registerEvent("onMissionStarted")
	AdditionalSettingsUtil.registerEvent("onDelete")
	AdditionalSettingsUtil.registerEvent("onUpdate")
	AdditionalSettingsUtil.registerEvent("onDraw")

	addModEventListener(self)
	g_currentMission:registerToLoadOnMapFinished(self)
	g_currentMission:registerObjectToCallOnMissionStart(self)

	local settingsDirectory = getUserProfileAppPath() .. "modsSettings/"
	local settingsXMLPath = settingsDirectory .. "/additionalSettings.xml"

	createFolder(settingsDirectory)

	self.modVersion = g_modManager:getModByName(modName).version
	self.settingsDirectory = settingsDirectory
	self.settingsXMLPath = settingsXMLPath
	self.baseDirectory = modDir
	self.resetAllowed = false
	self.addittionalSettings = self.loadAdditionalSettings()
	self.addModEnvironmentTexts(true)
	self.addSettingsPage(modDir)

	return self
end

function AdditionalSettingsManager.loadAdditionalSettings()
	return {
		hud = HUDSettings:new(),
		crosshair = CrosshairSettings:new(),
		currentDate = CurrentDateSettings:new(),
		clockPosition = ClockPositionSettings:new(),
		hourFormat = HourFormatSettings:new(),
		fadeEffect = FadeEffectSettings:new(),
		dialogBoxes = DialogBoxesSettings:new(),
		fillLevelHud = FillLevelHUDSettings:new(),
		vehicleCameraMovement = VehicleCameraMovementSettings:new(),
		playerCameraMovement = PlayerCameraMovementSettings:new(),
		easyMotorStart = EasyMotorStartSettings:new(),
		autostart = AutostartSettings:new(),
		storeItems = StoreItemsVisibilitySettings:new(),
		lighting = LightingSettings:new(),
		brightness = BrightnessSettings:new(),
		dof = DOFSettings:new(),
		birds = BirdsSettings:new(),
		animalsCollisions = AnimalsCollisionsSettings:new(),
		cameraCollisions = CameraCollisionsSettings:new(),
		guiCamera = GuiCameraSettings:new()
	}
end

function AdditionalSettingsManager.addModEnvironmentTexts(addText)
	local global_i18n = getfenv(0).g_i18n

	for name, value in pairs(g_i18n.texts) do
		if not addText then
			value = nil
		end

		global_i18n:setText(name, value)
	end
end

function AdditionalSettingsManager.addSettingsPage(baseDirectory)
	local additionalSettingsFrame = AdditionalSettingsFrame:new()
	local additionalSettingsFrameReference = AdditionalSettingsFrameReference:new()

	g_gui:loadGui(Utils.getFilename("xml/gui/AdditionalSettingsFrame.xml", baseDirectory), "AdditionalSettingsFrame", additionalSettingsFrame, true)
	g_gui:loadGui(Utils.getFilename("xml/gui/AdditionalSettingsFrameReference.xml", baseDirectory), "AdditionalSettingsFrameReference", additionalSettingsFrameReference)

	local inGameMenu = g_currentMission.inGameMenu
	local pageAdditionalSettings = additionalSettingsFrameReference.pageAdditionalSettings

	if pageAdditionalSettings ~= nil then
		local pagingElement = inGameMenu.pagingElement
		local index = pagingElement:getPageIndexByElement(inGameMenu.pageSettingsGeneral) + 1

		PagingElement:superClass().addElement(pagingElement, pageAdditionalSettings)
		pagingElement:addPage(string.upper(pageAdditionalSettings.name), pageAdditionalSettings, g_i18n:getText("additionalSettings_ui_ingameMenuAdditionalSettings"), index)

		inGameMenu:registerPage(pageAdditionalSettings, index, inGameMenu:makeIsGeneralSettingsEnabledPredicate())
		inGameMenu:addPageTab(pageAdditionalSettings, g_baseUIFilename, getNormalizedUVs({455, 144, -65, 65}))
		inGameMenu.pageAdditionalSettings = pageAdditionalSettings
	end
end

function AdditionalSettingsManager.addQuitGameButton()
	local inGameMenu = g_currentMission.inGameMenu
	local pageSettingsGame = inGameMenu.pageSettingsGame

	local function onButtonQuitGame()
		if inGameMenu.isSaving then
			return
		end

		if not inGameMenu.playerAlreadySaved and not (inGameMenu.missionDynamicInfo.isMultiplayer and inGameMenu.missionDynamicInfo.isClient) then
			g_gui:showYesNoDialog({
				text = g_i18n:getText(InGameMenu.L10N_SYMBOL.END_WITHOUT_SAVING),
				callback = function(yes)
					if yes then
						if inGameMenu.missionDynamicInfo.isMultiplayer and inGameMenu.isServer then
							inGameMenu.server:broadcastEvent(ShutdownEvent:new())
						end

						doExit()
					end
				end
			})
		else
			doExit()
		end
	end

	inGameMenu.quitGameButton = {
		showWhenPaused = true,
		inputAction = InputAction.MENU_EXTRA_2,
		text = g_i18n:getText("additionalSettings_button_quitGame"),
		callback = onButtonQuitGame
	}

	table.insert(inGameMenu.defaultMenuButtonInfo, inGameMenu.quitGameButton)
	inGameMenu.defaultMenuButtonInfoByActions[InputAction.MENU_EXTRA_2] = inGameMenu.quitGameButton
	inGameMenu.defaultButtonActionCallbacks[InputAction.MENU_EXTRA_2] = onButtonQuitGame

	pageSettingsGame.updateButtons = Utils.overwrittenFunction(pageSettingsGame.updateButtons, function(object, superFunc)
		object.menuButtonInfo = {
			object.backButtonInfo,
			object.saveButton,
			object.quitButton,
			inGameMenu.quitGameButton
		}

		if object.hasMasterRights and g_currentMission.missionDynamicInfo.isMultiplayer then
			table.insert(object.menuButtonInfo, object.serverSettingsButton)
		end

		object:setMenuButtonInfoDirty()
	end)

	pageSettingsGame:updateButtons()
end

function AdditionalSettingsManager:loadMap(filename)
	AdditionalSettingsUtil.appendedFunction(FSCareerMissionInfo, "saveToXMLFile", self, "saveToXMLFile")
	AdditionalSettingsUtil.raiseEvent("onLoad", filename)

	self.savedStates = self:loadSettingsFromXMLFile()
	self:applyXMLSettings(AdditionalSettingsManager.LOAD_STATE.LOAD_MAP)

	local pageAdditionalSettings = g_currentMission.inGameMenu.pageAdditionalSettings

	if pageAdditionalSettings ~= nil then
		pageAdditionalSettings:initialize(self)
	end

	self.addQuitGameButton()
end

function AdditionalSettingsManager:onLoadMapFinished()
	AdditionalSettingsUtil.raiseEvent("onLoadFinished")
	self:applyXMLSettings(AdditionalSettingsManager.LOAD_STATE.LOAD_MAP_FINISHED)
end

function AdditionalSettingsManager:onMissionStarted()
	AdditionalSettingsUtil.raiseEvent("onMissionStarted")
	self:applyXMLSettings(AdditionalSettingsManager.LOAD_STATE.MISSION_START)

	self.savedStates = {}
end

function AdditionalSettingsManager:deleteMap()
	AdditionalSettingsUtil.raiseEvent("onDelete")

	self.addModEnvironmentTexts(false)
	removeModEventListener(self)
	getfenv(0).g_additionalSettings = nil
end

function AdditionalSettingsManager:update(dt)
	AdditionalSettingsUtil.raiseEvent("onUpdate", dt)
end

function AdditionalSettingsManager:draw()
	AdditionalSettingsUtil.raiseEvent("onDraw")
end

function AdditionalSettingsManager:getSettingByName(setting)
	return self.addittionalSettings[setting]
end

function AdditionalSettingsManager:getSettingStateByName(setting)
	return self.addittionalSettings[setting].active or self.addittionalSettings[setting].state
end

function AdditionalSettingsManager.getSettingElement(setting)
	local pageAdditionalSettings = g_currentMission.inGameMenu.pageAdditionalSettings

	if pageAdditionalSettings ~= nil then
		for checkboxElement, settingsKey in pairs(pageAdditionalSettings.checkboxMapping) do
			if settingsKey == setting then
				return checkboxElement
			end
		end

		for optionElement, settingsKey in pairs(pageAdditionalSettings.optionMapping) do
			if settingsKey == setting then
				return optionElement
			end
		end
	end
end

function AdditionalSettingsManager.saveToXMLFile(settingsManager, object)
	settingsManager:saveSettingsToXMLFile()
end

function AdditionalSettingsManager:saveSettingsToXMLFile()
	local xmlFile = createXMLFile("additionalSettings", self.settingsXMLPath, "additionalSettings")

	if xmlFile ~= nil then
		setXMLString(xmlFile, "additionalSettings#version", self.modVersion)

		for _, setting in pairs(self.addittionalSettings) do
			if setting.name == "lighting" then
				if setting.state ~= 0 then
					setXMLString(xmlFile, "additionalSettings.lighting", HTMLUtil.encodeToHTML(setting.customLighting[setting.state].name))
				end
			else
				if setting.active ~= nil then
					setXMLBool(xmlFile, "additionalSettings." .. setting.name, setting.active)
				elseif setting.state ~= nil then
					setXMLInt(xmlFile, "additionalSettings." .. setting.name, setting.state)
				end
			end
		end

		saveXMLFile(xmlFile)
		delete(xmlFile)
	end
end

function AdditionalSettingsManager:loadSettingsFromXMLFile()
	local savedStates = {}

	if fileExists(self.settingsXMLPath) then
		local xmlFile = loadXMLFile("additionalSettings", self.settingsXMLPath)

		if xmlFile ~= nil then
			if self.resetAllowed and self.modVersion ~= getXMLString(xmlFile, "additionalSettings#version") then
				g_logManager:info("[AdditionalSettings] mod update detected, settings will be reset.")
			else
				for _, setting in pairs(self.addittionalSettings) do
					if setting.name == "lighting" then
						local lightingName = getXMLString(xmlFile, "additionalSettings.lighting")
						if lightingName ~= nil then
							savedStates[setting.name] = lightingName
						end
					else
						if setting.active ~= nil then
							local active = getXMLBool(xmlFile, "additionalSettings." .. setting.name)
							if active ~= nil and setting.active ~= active then
								savedStates[setting.name] = active
							end
						elseif setting.state ~= nil then
							local state = getXMLInt(xmlFile, "additionalSettings." .. setting.name)
							if state ~= nil and setting.state ~= state then
								savedStates[setting.name] = state
							end
						end
					end
				end
			end

			delete(xmlFile)
		end
	end

	return savedStates
end

function AdditionalSettingsManager:applyXMLSettings(loadState)
	for _, setting in pairs(self.addittionalSettings) do
		if setting.loadState == loadState then
			local savedSettingState = self.savedStates[setting.name]

			if savedSettingState ~= nil then
				if setting.name == "lighting" then
					local state = self:getSettingByName("lighting"):getLightingStateByName(savedSettingState)

					if state ~= nil and state ~= 0 then
						setting.state = state
						AdditionalSettingsUtil.callFunction(setting, "onStateChange", state, self.getSettingElement(setting), true)
					end
				else
					if setting.active ~= nil and setting.active ~= savedSettingState then
						setting.active = savedSettingState
						AdditionalSettingsUtil.callFunction(setting, "onStateChange", savedSettingState, self.getSettingElement(setting), true)
					elseif setting.state ~= nil and setting.state ~= savedSettingState then
						setting.state = savedSettingState
						AdditionalSettingsUtil.callFunction(setting, "onStateChange", savedSettingState, self.getSettingElement(setting), true)
					end
				end
			end
		end
	end
end