--
-- HUDSettings
--

HUDSettings = {}

function HUDSettings:new()
	self.name = "hud"
	self.active = true
	self.loadState = AdditionalSettingsManager.LOAD_STATE.MISSION_START

	return self
end

function HUDSettings:onStateChange(state, checkboxElement, loadFromSavegame)
	g_currentMission.hud:setIsVisible(state)

	if checkboxElement ~= nil then
		local target = checkboxElement.target

		if target ~= nil then
			target.multiCrosshair:setDisabled(not state or g_flightAndNoHUDKeysEnabled)
			target.multiCurrentDate:setDisabled(not state)
			target.multiClockPosition:setDisabled(not state or g_additionalSettingsManager:getSettingStateByName("currentDate") == 0)
		end
	end
end

function HUDSettings:onFrameOpen(checkboxElement)
	self.active = g_currentMission.hud:getIsVisible()
end

--
-- CrosshairSettings
--

CrosshairSettings = {}

function CrosshairSettings:new()
	AdditionalSettingsUtil.registerEventListener("onMissionStarted", self)

	self.name = "crosshair"
	self.state = 0
	self.loadState = AdditionalSettingsManager.LOAD_STATE.MISSION_START

	return self
end

function CrosshairSettings:onMissionStarted()
	local player = g_currentMission.player

	if player ~= nil then
		self.pickedUpObjectAimingUVs = player.pickedUpObjectAimingUVs
		self.simpleICAvailable = g_modIsLoaded["FS19_simpleIC"]
		AdditionalSettingsUtil.prependedFunction(player.pickedUpObjectOverlay, "render", self, "render")
	end
end

function CrosshairSettings:onFrameOpen(optionElement)
	optionElement:setDisabled(not g_currentMission.hud:getIsVisible() or g_flightAndNoHUDKeysEnabled)
end

function CrosshairSettings.render(setting, object)
	local isVisible = true

	if setting.state ~= 0 then
		if setting.state == 2 or (setting.state == 1 and object.uvs == setting.pickedUpObjectAimingUVs and not setting:getIsICActive()) then
			isVisible = false
		end
	end

	object:setIsVisible(isVisible)
end

function CrosshairSettings:getIsICActive()
	if self.simpleICAvailable then
		for _, displayActionEvent in ipairs(g_inputBinding:getDisplayActionEvents()) do
			if displayActionEvent.action.name == "INTERACT_IC_ONFOOT" then
				return true
			end
		end
	end

	return false
end

--
-- CurrentDateSettings
--

CurrentDateSettings = {}

function CurrentDateSettings:new()
	AdditionalSettingsUtil.registerEventListener("onLoad", self)

	self.name = "currentDate"
	self.state = 0
	self.loadState = AdditionalSettingsManager.LOAD_STATE.MISSION_START

	self.dateFormat = ""

	self.blinking = true
	self.blinkTime = 0
	self.blinkSec = -1

	self.clockSettings = {
		[0] = {horizontalAlignment = RenderText.ALIGN_RIGHT, verticalAligment = RenderText.VERTICAL_ALIGN_MIDDLE, posX = 1 - g_safeFrameOffsetX, posY = 1 - g_safeFrameOffsetY / 2, textColor = {1, 1, 1, 1}, textBold = false, hasShadow = false},
		[1] = {horizontalAlignment = RenderText.ALIGN_RIGHT, verticalAligment = RenderText.VERTICAL_ALIGN_MIDDLE, posX = 1 - g_safeFrameOffsetX, posY = g_safeFrameOffsetY / 2, textColor = {1, 1, 1, 1}, textBold = false, hasShadow = false},
		[2] = {horizontalAlignment = RenderText.ALIGN_LEFT, verticalAligment = RenderText.VERTICAL_ALIGN_MIDDLE, posX = g_safeFrameOffsetX, posY = 1 - g_safeFrameOffsetY / 2, textColor = {1, 1, 1, 1}, textBold = false, hasShadow = false},
		[3] = {horizontalAlignment = RenderText.ALIGN_LEFT, verticalAligment = RenderText.VERTICAL_ALIGN_MIDDLE, posX = g_safeFrameOffsetX, posY = g_safeFrameOffsetY / 2, textColor = {1, 1, 1, 1}, textBold = false, hasShadow = false}
	}

	self.dayName = {
		["Monday"] = g_i18n:getText("ui_financesDay1"),
		["Tuesday"] = g_i18n:getText("ui_financesDay2"),
		["Wednesday"] = g_i18n:getText("ui_financesDay3"),
		["Thursday"] = g_i18n:getText("ui_financesDay4"),
		["Friday"] = g_i18n:getText("ui_financesDay5"),
		["Saturday"] = g_i18n:getText("ui_financesDay6"),
		["Sunday"] = g_i18n:getText("ui_financesDay7")
	}

	self.monthName = {
		["January"] = g_i18n:getText("additionalSettings_ui_monthJanuary"),
		["February"] = g_i18n:getText("additionalSettings_ui_monthFebruary"),
		["March"] = g_i18n:getText("additionalSettings_ui_monthMarch"),
		["April"] = g_i18n:getText("additionalSettings_ui_monthApril"),
		["May"] = g_i18n:getText("additionalSettings_ui_monthMay"),
		["June"] = g_i18n:getText("additionalSettings_ui_monthJune"),
		["July"] = g_i18n:getText("additionalSettings_ui_monthJuly"),
		["August"] = g_i18n:getText("additionalSettings_ui_monthAugust"),
		["September"] = g_i18n:getText("additionalSettings_ui_monthSeptember"),
		["October"] = g_i18n:getText("additionalSettings_ui_monthOctober"),
		["November"] = g_i18n:getText("additionalSettings_ui_monthNovember"),
		["December"] = g_i18n:getText("additionalSettings_ui_monthDecember")
	}

	local function dateSubstr(text, indexOfFirstCharacter, indexOfLastCharacter, dot)
		local trimText = utf8Substr(text, indexOfFirstCharacter, indexOfLastCharacter)

		if dot and utf8Strlen(text) > indexOfLastCharacter then
			trimText = trimText .. "."
		end

		return trimText
	end

	self.dayNameShort = {}

	for dayName, text in pairs(self.dayName) do
		self.dayNameShort[dayName] = dateSubstr(text, 0, 3, false)
	end

	self.monthNameShort = {}

	for monthName, text in pairs(self.monthName) do
		self.monthNameShort[monthName] = dateSubstr(text, 0, 3, true)
	end

	return self
end

function CurrentDateSettings:onLoad(filename)
	AdditionalSettingsUtil.prependedFunction(g_currentMission.hud.gameInfoDisplay, "drawTimeText", self, "drawTimeText")
end

function CurrentDateSettings:onStateChange(state, optionElement, loadFromSavegame)
	self:updateCurrentDateFormat(state, g_additionalSettingsManager:getSettingStateByName("hourFormat"))

	if optionElement ~= nil and optionElement.target ~= nil then
		optionElement.target.multiClockPosition:setDisabled(not g_currentMission.hud:getIsVisible() or state == 0)
	end
end

function CurrentDateSettings:onCreateElement(optionElement)
	optionElement:setTexts({
		g_i18n:getText("ui_off"),
		g_i18n:getText("additionalSettings_ui_currentTimeHour"),
		string.format("%s + %s", g_i18n:getText("additionalSettings_ui_currentTimeHour"), g_i18n:getText("additionalSettings_ui_currentTimeDate")),
		string.format("%s + %s (%s)", g_i18n:getText("additionalSettings_ui_currentTimeHour"), g_i18n:getText("additionalSettings_ui_currentTimeDate"), g_i18n:getText("additionalSettings_ui_currentTimeDateShort")),
		string.format("%s + %s (%s)", g_i18n:getText("additionalSettings_ui_currentTimeHour"), g_i18n:getText("additionalSettings_ui_currentTimeDate"), g_i18n:getText("additionalSettings_ui_currentTimeDateLong"))
	})
end

function CurrentDateSettings:onFrameOpen(optionElement)
	optionElement:setDisabled(not g_currentMission.hud:getIsVisible())
end

function CurrentDateSettings:updateCurrentDateFormat(state, hourFormat)
	local hour = "%H"
	local am_pm = ""

	if not hourFormat then
		hour = "%I"
		am_pm = " %p"
	end

	local dateFormat = hour .. ":%M:%S" .. am_pm

	if state == 2 then
		dateFormat = dateFormat .. " | " .. g_i18n:getCurrentDate()
	elseif state == 3 or state == 4 then
		dateFormat = dateFormat .. " | dayName, monthDay monthName %Y"
	end

	self.dateFormat = dateFormat
end

function CurrentDateSettings.drawTimeText(setting, object)
	if setting.state ~= 0 then
		local dateFormat = setting.dateFormat

		if setting.state == 3 or setting.state == 4 then
			local monthDay = string.gsub(getDate("%d"), "^0+", "")
			local dayName = getDate("%A")
			local monthName = getDate("%B")
			local dayNameM = setting.dayName[dayName]
			local monthNameM = setting.monthName[monthName]

			if setting.state == 3 then
				dayNameM = setting.dayNameShort[dayName]
				monthNameM = setting.monthNameShort[monthName]
			end

			dateFormat = string.gsub(dateFormat, "monthDay", monthDay)
			dateFormat = string.gsub(dateFormat, "dayName", dayNameM or dayName)
			dateFormat = string.gsub(dateFormat, "monthName", monthNameM or monthName)
		end

		dateFormat = getDate(dateFormat)

		if setting.blinking then
			local seconds = tonumber(dateFormat:sub(7, 8))

			if seconds ~= nil then
				if seconds ~= setting.blinkSec then
					setting.blinkSec = seconds
					setting.blinkTime = 0
				else
					setting.blinkTime = setting.blinkTime + g_currentDt
				end

				if setting.blinkTime <= 250 or setting.blinkTime >= 750 then
					dateFormat = string.gsub(dateFormat, ":", " ")
				end
			end
		end

		local clockSettings = setting.clockSettings[g_additionalSettingsManager:getSettingStateByName("clockPosition")]
		local textSize = object.moneyTextSize * 0.8

		setTextBold(clockSettings.textBold)
		setTextAlignment(clockSettings.horizontalAlignment)
		setTextVerticalAlignment(clockSettings.verticalAligment)

		if clockSettings.hasShadow then
			local offset = textSize * HUDTextDisplay.SHADOW_OFFSET_FACTOR

			setTextColor(0, 0, 0, 1)
			renderText(clockSettings.posX + offset, clockSettings.posY - offset, textSize, dateFormat)
		end

		setTextColor(unpack(clockSettings.textColor))
		renderText(clockSettings.posX, clockSettings.posY, textSize, dateFormat)
		setTextAlignment(RenderText.ALIGN_LEFT)
		setTextVerticalAlignment(RenderText.VERTICAL_ALIGN_BASELINE)
		setTextColor(1, 1, 1, 1)
	end
end

--
-- ClockPositionSettings
--

ClockPositionSettings = {}

function ClockPositionSettings:new()
	self.name = "clockPosition"
	self.state = 0
	self.loadState = AdditionalSettingsManager.LOAD_STATE.LOAD_MAP

	return self
end

function ClockPositionSettings:onFrameOpen(optionElement)
	optionElement:setDisabled(not g_currentMission.hud:getIsVisible() or g_additionalSettingsManager:getSettingStateByName("currentDate") == 0)
end

--
-- HourFormatSettings
--

HourFormatSettings = {}

function HourFormatSettings:new()
	AdditionalSettingsUtil.registerEventListener("onLoad", self)

	self.name = "hourFormat"
	self.active = true
	self.loadState = AdditionalSettingsManager.LOAD_STATE.LOAD_MAP

	return self
end

function HourFormatSettings:onLoad(filename)
	AdditionalSettingsUtil.overwrittenFunction(g_currentMission.hud.gameInfoDisplay, "updateTime", self, "updateTime")
end

function HourFormatSettings:onStateChange(state, checkboxElement, loadFromSavegame)
	local gameInfoDisplay = g_currentMission.hud.gameInfoDisplay
	local timeBoxWidth = GameInfoDisplay.SIZE.TIME_BOX[1]

	if not state then
		timeBoxWidth = GameInfoDisplay.SIZE.TIME_BOX[1] + 37.5
	end

	local boxWidth = gameInfoDisplay:scalePixelToScreenWidth(timeBoxWidth)
	local sepX, sepY = gameInfoDisplay.timeBox.separator:getPosition()
	local arrowX, arrowY = gameInfoDisplay.timeScaleArrow:getPosition()
	local fArrowX, fArrowY = gameInfoDisplay.timeScaleArrowFast:getPosition()

	gameInfoDisplay.timeBox.overlay.width = boxWidth

	gameInfoDisplay:storeScaledValues()
	gameInfoDisplay:updateSizeAndPositions()

	gameInfoDisplay.timeBox.separator:setPosition(sepX, sepY)
	gameInfoDisplay.timeScaleArrow:setPosition(arrowX, arrowY)
	gameInfoDisplay.timeScaleArrowFast:setPosition(fArrowX, fArrowY)

	local currentDateSettings = g_additionalSettingsManager:getSettingByName("currentDate")
	currentDateSettings:updateCurrentDateFormat(currentDateSettings.state, state)
end

function HourFormatSettings.updateTime(setting, object, superFunc)
	local currentTime = object.environment.dayTime / 3600000
	local timeHours = math.floor(currentTime)
	local timeMinutes = math.floor((currentTime - timeHours) * 60)

	local timeHoursText = timeHours
	local am_pm = ""

	if not setting.active then
		am_pm = " AM"

		if timeHoursText == 0 then
			timeHoursText = 12
		elseif timeHoursText > 11 then
			if timeHoursText > 12 then
				timeHoursText = timeHoursText - 12
			end

			am_pm = " PM"
		end
	end

	object.timeText = string.format("%02d:%02d%s", timeHoursText, timeMinutes, am_pm)
	object.timeScaleText = string.format("%d", object.missionInfo.timeScale)

	local hourRotation = -(currentTime % 12 / 12) * math.pi * 2
	local minutesRotation = -(currentTime - timeHours) * math.pi * 2

	object.clockHandSmall:setRotation(hourRotation)
	object.clockHandLarge:setRotation(minutesRotation)

	local isTimeScaleFast = object.missionInfo.timeScale > 1

	object.timeScaleArrow:setVisible(not isTimeScaleFast)
	object.timeScaleArrowFast:setVisible(isTimeScaleFast)
end

--
-- FadeEffectSettings
--

FadeEffectSettings = {}

function FadeEffectSettings:new()
	AdditionalSettingsUtil.registerEventListener("onUpdate", self)
	AdditionalSettingsUtil.registerEventListener("onDraw", self)
	AdditionalSettingsUtil.overwrittenStaticFunction(getfenv(0), "setCamera", self, "setCamera")
	AdditionalSettingsUtil.overwrittenFunction(g_gui, "changeScreen", self, "changeScreen")

	self.name = "fadeEffect"
	self.state = 0
	self.loadState = AdditionalSettingsManager.LOAD_STATE.MISSION_START

	local fadeOverlay = Overlay:new(g_baseHUDFilename, 0, 0, 1, 1)
	fadeOverlay:setUVs(getNormalizedUVs(HUD.UV.AREA))
	fadeOverlay:setColor(0, 0, 0, 0)

	self.fadeScreenElement = HUDElement:new(fadeOverlay)
	self.fadeAnimation = TweenSequence.NO_SEQUENCE
	self.effectDuration = 0

	return self
end

function FadeEffectSettings:onUpdate(dt)
	if g_currentMission:getIsClient() then
		if not self.fadeAnimation:getFinished() then
			self.fadeAnimation:update(dt)
		end
	end
end

function FadeEffectSettings:onDraw()
	if self.fadeScreenElement:getVisible() then
		self.fadeScreenElement:draw()
	end
end

function FadeEffectSettings:onCreateElement(optionElement)
	local texts = {
		g_i18n:getText("ui_off")
	}

	for i = 100, 1000 + 0.0001, 100 do
		table.insert(texts, string.format("%d ms", i))
	end

	optionElement:setTexts(texts)
end

function FadeEffectSettings:getFadeEffectDurationFromIndex(index)
	return MathUtil.clamp(index * 100, 0, 1000)
end

function FadeEffectSettings:onStateChange(state, optionElement, loadFromSavegame)
	self.effectDuration = self:getFadeEffectDurationFromIndex(state)
end

function FadeEffectSettings:fadeScreen()
	if not g_currentMission.hud:getIsFading() and self.effectDuration ~= 0 then
		local seq = TweenSequence.new(self.fadeScreenElement)
		seq:addTween(Tween:new(self.fadeScreenElement.setAlpha, 1, 0, self.effectDuration))
		seq:start()

		self.fadeAnimation = seq
	end
end

function FadeEffectSettings.setCamera(setting, superFunc, camera)
	local disableFadeEffect = false
	local player = g_currentMission.player

	if player ~= nil then
		disableFadeEffect = camera == player.cuttingCameraNode or getCamera() == player.cuttingCameraNode
	end

	if not disableFadeEffect then
		setting:fadeScreen()
	end

	superFunc(camera)
end

function FadeEffectSettings.changeScreen(setting, object, superFunc, source, screenClass, returnScreenClass)
	local disableFadeEffect = object.currentGuiName == "ChatDialog"
	local retValue = superFunc(object, source, screenClass, returnScreenClass)

	if not disableFadeEffect and object.currentGuiName == "" then
		setting:fadeScreen()
	end

	return retValue
end

--
-- DialogBoxesSettings
--

DialogBoxesSettings = {}

function DialogBoxesSettings:new()
	AdditionalSettingsUtil.registerEventListener("onUpdate", self)
	AdditionalSettingsUtil.overwrittenFunction(g_gui, "showInfoDialog", self, "showInfoDialog")
	AdditionalSettingsUtil.overwrittenFunction(g_gui, "showMessageDialog", self, "showMessageDialog")

	self.name = "dialogBoxes"
	self.active = true
	self.loadState = AdditionalSettingsManager.LOAD_STATE.LOAD_MAP

	self.infoDialogs = {
		{text = "ui_vehicleResetDone", callback = true},
		{text = "shop_messagePurchaseReady", callback = true},
		{text = "shop_messageLeasingReady", callback = true},
		{text = "shop_messageSoldVehicle", callback = true},
		{text = "shop_messageBoughtChainsaw", callback = true},
		{text = "shop_messageReturnedVehicle", callback = true}
	}

	self.messageDialogs = {
		"shop_messageBuyingVehicle",
		"shop_messageLeasingVehicle"
	}

	self.callbacks = {}
	self.inputDelay = 0

	return self
end

function DialogBoxesSettings:onUpdate(dt)
	if g_currentMission:getIsClient() then
		for callback in pairs(self.callbacks) do
			if self.inputDelay < g_currentMission.time then
				self.callbacks[callback]()
				self.callbacks[callback] = nil
			end
		end
	end
end

function DialogBoxesSettings.showInfoDialog(setting, object, superFunc, args)
	local showInfoDialog = true

	if not setting.active then
		if args ~= nil then
			for _, infoDialog in pairs(setting.infoDialogs) do
				if args.text == g_i18n:getText(infoDialog.text) then
					if infoDialog.callback then
						local function callback()
							if args.target ~= nil then
								args.callback(args.target, args.args)
							else
								args.callback(args.args)
							end
						end

						setting.inputDelay = g_currentMission.time + 125
						setting.callbacks[callback] = callback
					end

					showInfoDialog = false
					break
				end
			end
		end
	end

	if showInfoDialog then
		superFunc(object, args)
	end
end

function DialogBoxesSettings.showMessageDialog(setting, object, superFunc, args)
	local showMessageDialog = true

	if not setting.active then
		if args ~= nil then
			for _, dialogText in pairs(setting.messageDialogs) do
				if args.text == g_i18n:getText(dialogText) then
					showMessageDialog = false
					break
				end
			end
		end
	end

	if showMessageDialog then
		superFunc(object, args)
	end
end

--
-- FillLevelHUDSettings
--

FillLevelHUDSettings = {}

function FillLevelHUDSettings:new()
	AdditionalSettingsUtil.registerEventListener("onDraw", self)

	self.name = "fillLevelHud"
	self.active = false
	self.loadState = AdditionalSettingsManager.LOAD_STATE.LOAD_MAP

	self.huds = {}

	return self
end

function FillLevelHUDSettings:loadHUD()
	local hud = TrailerFillLevelHUD.new(g_baseHUDFilename)
	self.huds[hud] = hud

	return hud
end

function FillLevelHUDSettings:deleteHUD(hud)
	self.huds[hud] = nil
	hud:delete()
end

function FillLevelHUDSettings:onDraw()
	for k in pairs(self.huds) do
		self.huds[k]:draw()
	end
end

--
-- VehicleCameraMovementSettings
--

VehicleCameraMovementSettings = {}

function VehicleCameraMovementSettings:new()
	AdditionalSettingsUtil.registerEventListener("onLoad", self)

	self.name = "vehicleCameraMovement"
	self.state = 0
	self.loadState = AdditionalSettingsManager.LOAD_STATE.MISSION_START

	self.yaw = 0
	self.pitch = 0
	self.value = 0

	return self
end

function VehicleCameraMovementSettings:onLoad(filename)
	AdditionalSettingsUtil.overwrittenFunction(VehicleCamera, "actionEventLookLeftRight", self, "actionEventLookLeftRight")
	AdditionalSettingsUtil.overwrittenFunction(VehicleCamera, "actionEventLookUpDown", self, "actionEventLookUpDown")
end

function VehicleCameraMovementSettings:onCreateElement(optionElement)
	local texts = {
		g_i18n:getText("ui_off")
	}

	for i = 10, 100 + 0.0001, 10 do
		table.insert(texts, string.format("%d%%", i))
	end

	optionElement:setTexts(texts)
end

function VehicleCameraMovementSettings:getSmoothCameraValueFromIndex(index)
	return MathUtil.clamp(5 - index * 0.35, 1.5, 5)
end

function VehicleCameraMovementSettings:onStateChange(state, optionElement, loadFromSavegame)
	self.value = self:getSmoothCameraValueFromIndex(state)
end

function VehicleCameraMovementSettings:getSmoothValue(pitch, inputValue, isMouse)
	if isMouse then
		inputValue = inputValue * 0.001 * 16.666
	else
		inputValue = inputValue * 0.001 * g_currentDt
	end

	self[pitch] = inputValue + math.pow(0.99579, g_currentDt * self.value) * (self[pitch] - inputValue)

	return self[pitch]
end

function VehicleCameraMovementSettings.actionEventLookLeftRight(setting, object, superFunc, actionName, inputValue, callbackState, isAnalog, isMouse)
	if setting.state == 0 then
		superFunc(object, actionName, inputValue, callbackState, isAnalog, isMouse)
	else
		object.lastInputValues.leftRight = object.lastInputValues.leftRight + setting:getSmoothValue("yaw", inputValue, isMouse)
	end
end

function VehicleCameraMovementSettings.actionEventLookUpDown(setting, object, superFunc, actionName, inputValue, callbackState, isAnalog, isMouse)
	if setting.state == 0 then
		superFunc(object, actionName, inputValue, callbackState, isAnalog, isMouse)
	else
		object.lastInputValues.upDown = object.lastInputValues.upDown + setting:getSmoothValue("pitch", inputValue, isMouse)
	end
end

--
-- PlayerCameraMovementSettings
--

PlayerCameraMovementSettings = {}

function PlayerCameraMovementSettings:new()
	AdditionalSettingsUtil.registerEventListener("onLoad", self)

	self.name = "playerCameraMovement"
	self.state = 0
	self.loadState = AdditionalSettingsManager.LOAD_STATE.MISSION_START

	self.yaw = 0
	self.pitch = 0
	self.value = 0

	return self
end

function PlayerCameraMovementSettings:onLoad(filename)
	AdditionalSettingsUtil.overwrittenFunction(Player, "onInputLookLeftRight", self, "onInputLookLeftRight")
	AdditionalSettingsUtil.overwrittenFunction(Player, "onInputLookUpDown", self, "onInputLookUpDown")
end

function PlayerCameraMovementSettings:onCreateElement(optionElement)
	local texts = {
		g_i18n:getText("ui_off")
	}

	for i = 10, 100 + 0.0001, 10 do
		table.insert(texts, string.format("%d%%", i))
	end

	optionElement:setTexts(texts)
end

function PlayerCameraMovementSettings:getSmoothCameraValueFromIndex(index)
	return MathUtil.clamp(5 - index * 0.35, 1.5, 5)
end

function PlayerCameraMovementSettings:onStateChange(state, optionElement, loadFromSavegame)
	self.value = self:getSmoothCameraValueFromIndex(state)
end

function PlayerCameraMovementSettings:getSmoothValue(pitch, inputValue, isMouse)
	if isMouse then
		inputValue = inputValue * 0.001 * 16.666
	else
		inputValue = inputValue * 0.001 * g_currentDt
	end

	self[pitch] = inputValue + math.pow(0.99579, g_currentDt * self.value) * (self[pitch] - inputValue)

	return self[pitch]
end

function PlayerCameraMovementSettings.onInputLookLeftRight(setting, object, superFunc, actionName, inputValue, callbackState, isAnalog, isMouse)
	if setting.state == 0 then
		superFunc(object, actionName, inputValue, callbackState, isAnalog, isMouse)
	else
		if not object.lockedInput then
			object.inputInformation.yawCamera = object.inputInformation.yawCamera + setting:getSmoothValue("yaw", inputValue, isMouse)
		end

		object.inputInformation.isMouseRotation = isMouse
	end
end

function PlayerCameraMovementSettings.onInputLookUpDown(setting, object, superFunc, actionName, inputValue, callbackState, isAnalog, isMouse)
	if setting.state == 0 then
		superFunc(object, actionName, inputValue, callbackState, isAnalog, isMouse)
	else
		if not object.lockedInput then
			local pitchValue = g_gameSettings:getValue("invertYLook") and -inputValue or inputValue

			object.inputInformation.pitchCamera = object.inputInformation.pitchCamera + setting:getSmoothValue("pitch", pitchValue, isMouse)
		end
	end
end

--
-- EasyMotorStartSettings
--

EasyMotorStartSettings = {}

function EasyMotorStartSettings:new()
	AdditionalSettingsUtil.appendedFunction(Drivable, "actionEventAccelerate", self, "actionEventAccelerate")

	self.name = "easyMotorStart"
	self.active = false
	self.loadState = AdditionalSettingsManager.LOAD_STATE.LOAD_MAP

	return self
end

function EasyMotorStartSettings:onFrameOpen(checkboxElement)
	checkboxElement:setDisabled(g_currentMission.missionInfo.automaticMotorStartEnabled)
end

function EasyMotorStartSettings.actionEventAccelerate(setting, object, actionName, inputValue, callbackState, isAnalog)
	if not g_currentMission.missionInfo.automaticMotorStartEnabled and setting.active then
		if inputValue > 0 then
			if not object:getIsAIActive() and not object:getIsMotorStarted() and object:getCanMotorRun() then
				object:startMotor()
			end
		end
	end
end

--
-- AutostartSettings
--

AutostartSettings = {}

function AutostartSettings:new()
	AdditionalSettingsUtil.appendedFunction(g_mpLoadingScreen, "onReadyToStart", self, "onReadyToStart")

	self.name = "autostart"
	self.active = false
	self.loadState = AdditionalSettingsManager.LOAD_STATE.LOAD_MAP

	return self
end

function AutostartSettings.onReadyToStart(setting, object)
	if setting.active and g_currentMission:canStartMission() and g_dedicatedServerInfo == nil and not g_gui:getIsDialogVisible() then
		object:onClickOk()
	end
end

--
-- StoreItemsVisibilitySettings
--

StoreItemsVisibilitySettings = {}

function StoreItemsVisibilitySettings:new()
	AdditionalSettingsUtil.overwrittenFunction(g_currentMission.shopMenu, "onClickBrand", self, "onClickBrand")
	AdditionalSettingsUtil.overwrittenFunction(g_currentMission.shopMenu, "onClickItemCategory", self, "onClickItemCategory")

	self.name = "storeItems"
	self.state = 0
	self.loadState = AdditionalSettingsManager.LOAD_STATE.MISSION_START

	return self
end

function StoreItemsVisibilitySettings:getIsItemVisible(item)
	local state = self.state

	if state == 0 or item == nil then
		return false
	end

	local isMod = false
	local isDlc = false

	if item.customEnvironment ~= nil then
		local isDLCEnv = StringUtil.startsWith(item.customEnvironment, g_uniqueDlcNamePrefix)

		isMod = not isDLCEnv
		isDlc = isDLCEnv
	end

	if state == 1 then
		return not isMod and not isDlc
	elseif state == 2 then
		return isMod or isDlc
	elseif state == 3 then
		return isMod
	else
		return isDlc
	end
end

function StoreItemsVisibilitySettings:onStateChange(state, optionElement, loadFromSavegame)
	local itemsCategoryName = ""

	if state ~= 0 and optionElement ~= nil then
		itemsCategoryName = string.format(" (%s)", optionElement.texts[state + 1])
	end

	local shopMenu = g_currentMission.shopMenu
	local shopPages = {
		[shopMenu.pageShopVehicles] = {
			shopMenu.shopController:getVehicleCategories(),
			shopMenu:makeSelfCallback(shopMenu.onClickItemCategory),
			getNormalizedUVs(ShopMenu.TAB_UV.VEHICLES),
			shopMenu.l10n:getText(ShopMenu.L10N_SYMBOL.HEADER_VEHICLES) .. itemsCategoryName
		},
		[shopMenu.pageShopTools] = {
			shopMenu.shopController:getToolCategories(),
			shopMenu:makeSelfCallback(shopMenu.onClickItemCategory),
			getNormalizedUVs(ShopMenu.TAB_UV.TOOLS),
			shopMenu.l10n:getText(ShopMenu.L10N_SYMBOL.HEADER_TOOLS) .. itemsCategoryName
		},
		[shopMenu.pageShopObjects] = {
			shopMenu.shopController:getObjectCategories(),
			shopMenu:makeSelfCallback(shopMenu.onClickItemCategory),
			getNormalizedUVs(ShopMenu.TAB_UV.OBJECTS),
			shopMenu.l10n:getText(ShopMenu.L10N_SYMBOL.HEADER_OBJECTS) .. itemsCategoryName
		},
		[shopMenu.pageShopPlaceables] = {
			shopMenu.shopController:getPlaceableCategories(),
			shopMenu:makeSelfCallback(shopMenu.onClickItemCategory),
			getNormalizedUVs(ShopMenu.TAB_UV.PLACEABLES),
			shopMenu.l10n:getText(ShopMenu.L10N_SYMBOL.HEADER_PLACEABLES) .. itemsCategoryName
		},
		[shopMenu.pageShopBrands] = {
			shopMenu.shopController:getBrands(),
			shopMenu:makeSelfCallback(shopMenu.onClickBrand),
			getNormalizedUVs(ShopMenu.TAB_UV.BRANDS),
			shopMenu.l10n:getText(ShopMenu.L10N_SYMBOL.HEADER_BRANDS)  .. itemsCategoryName,
			ShopMenu.BRAND_IMAGE_HEIGHT_WIDTH_RATIO
		}
	}

	for page, attr in pairs(shopPages) do
		if state ~= 0 then
			local categories = attr[1]
			local visibleCategories = {}

			for _, category in ipairs(categories) do
				local items = {}

				if page == shopMenu.pageShopBrands then
					items = shopMenu.shopController:getItemsByBrand(category.id)
				else
					items = shopMenu.shopController:getItemsByCategory(category.id)
				end

				local addCategory = false

				for i = 1, #items do
					if not addCategory and self:getIsItemVisible(items[i].storeItem) then
						table.insert(visibleCategories, category)
						addCategory = true
					end
				end
			end

			attr[1] = visibleCategories
		end

		page:reset()
		page:initialize(unpack(attr))
	end
end

function StoreItemsVisibilitySettings.onClickBrand(setting, object, superFunc, brandId, brandCategoryIconUVs, brandCategoryDisplayName, categoryDisplayName)
	local brandItems = object.shopController:getItemsByBrand(brandId)
	local currentDisplayItems = brandItems

	if setting.state ~= 0 then
		local displayItems = {}

		for i = 1, #brandItems do
			if setting:getIsItemVisible(brandItems[i].storeItem) then
				table.insert(displayItems, brandItems[i])
			end
		end

		currentDisplayItems = displayItems
	end

	object.currentDisplayItems = currentDisplayItems
	object.pageShopItemDetails:setDisplayItems(object.currentDisplayItems, false)
	object.pageShopItemDetails:setCategory(brandCategoryIconUVs, brandCategoryDisplayName, categoryDisplayName)
	object:pushDetail(object.pageShopItemDetails)
end

function StoreItemsVisibilitySettings.onClickItemCategory(setting, object, superFunc, categoryName, baseCategoryIconUVs, baseCategoryDisplayName, categoryDisplayName)
	local categoryItems = object.shopController:getItemsByCategory(categoryName)
	local currentDisplayItems = categoryItems

	if setting.state ~= 0 then
		local displayItems = {}

		for i = 1, #categoryItems do
			if setting:getIsItemVisible(categoryItems[i].storeItem) then
				table.insert(displayItems, categoryItems[i])
			end
		end

		currentDisplayItems = displayItems
	end

	object.currentDisplayItems = currentDisplayItems
	object.pageShopItemDetails:setDisplayItems(object.currentDisplayItems, false)
	object.pageShopItemDetails:setCategory(baseCategoryIconUVs, baseCategoryDisplayName, categoryDisplayName)
	object:pushDetail(object.pageShopItemDetails)
end

--
-- LightingSettings
--

LightingSettings = {}

function LightingSettings:new()
	AdditionalSettingsUtil.registerEventListener("onLoad", self)
	AdditionalSettingsUtil.registerEventListener("onDelete", self)

	self.name = "lighting"
	self.state = 0
	self.loadState = AdditionalSettingsManager.LOAD_STATE.MISSION_START

	return self
end

function LightingSettings:onLoad(filename)
	AdditionalSettingsUtil.appendedFunction(g_shopConfigScreen.elements[1], "onCloseCallback", self, "onCloseCallback")

	local lightingDirectory = g_additionalSettingsManager.settingsDirectory .. "lighting/"
	createFolder(lightingDirectory)

	self:copyBaseLightingFiles(g_additionalSettingsManager.baseDirectory .. "xml/lighting/", lightingDirectory .. "sample/")
	self.customLighting, self.customLightingTexts = self:loadCustomLightingConfigurations(lightingDirectory)
	self.lightingDirectory = lightingDirectory

	addConsoleCommand("gsReloadCustomLighting", "", "consoleCommandReloadCustomLighting", self)
end

function LightingSettings:onDelete()
	removeConsoleCommand("gsReloadCustomLighting")
end

function LightingSettings:onStateChange(state, optionElement, loadFromSavegame)
	local lighting = nil

	if state ~= nil and state ~= 0 then
		lighting = self.customLighting[state].lighting
	end

	g_currentMission.environment:setCustomLighting(lighting)
end

function LightingSettings:onCreateElement(optionElement)
	optionElement:setTexts(self.customLightingTexts)
	optionElement.toolTipText = string.format(optionElement.toolTipText, self.lightingDirectory)
end

function LightingSettings:copyBaseLightingFiles(directory, targetDirectory)
	createFolder(targetDirectory)
	copyFile(directory .. "lighting.xml", targetDirectory .. "lighting.xml", false)
	copyFile(directory .. "colorGrading.xml", targetDirectory .. "colorGrading.xml", false)
	copyFile(directory .. "colorGradingNight.xml", targetDirectory .. "colorGradingNight.xml", false)
end

function LightingSettings:loadCustomLightingConfigurations(directory)
	local customLighting = {}
	local customLightingTexts = {g_i18n:getText("ui_off")}
	local baseDirectoryBackup = g_currentMission.baseDirectory
	local files = Files:new(directory)

	for _, v in pairs(files.files) do
		if v.isDirectory then
			local baseDirectory = directory .. v.filename .. "/"
			local xmlFilename = baseDirectory .. "lighting.xml"

			if fileExists(xmlFilename) then
				local xmlFile = loadXMLFile("lightingXML", xmlFilename)

				if xmlFile ~= nil then
					g_currentMission.baseDirectory = baseDirectory

					local lighting = Lighting:new(g_currentMission.environment)
					lighting:load(xmlFile, "lighting")

					local name = v.filename
					local text = name

					if g_i18n:hasText("additionalSettings_customLighting_" .. name) then
						text = g_i18n:getText("additionalSettings_customLighting_" .. name)
					end

					table.insert(customLighting, {lighting = lighting, name = name})
					table.insert(customLightingTexts, text)

					g_logManager:info("Lighting configuration loaded: [%s] '%s'.", text, xmlFilename)

					delete(xmlFile)
				end
			else
				g_logManager:error("File '%s' not found!", xmlFilename)
			end
		end
	end

	for i = 1, g_mapManager:getNumOfMaps() do
		local map = g_mapManager:getMapDataByIndex(i)

		if map ~= g_currentMission.missionInfo.map then
			local mapXMLFilename = Utils.getFilename(map.mapXMLFilename, map.baseDirectory)
			local mapXML = loadXMLFile("MapXML", mapXMLFilename)

			if mapXML ~= nil then
				local environmentXMLFilename = Utils.getFilename(getXMLString(mapXML, "map.environment#filename"), map.baseDirectory)

				if environmentXMLFilename == "data/maps/mapDE_environment.xml" and map.title ~= g_i18n:getText("mapDE_title") or environmentXMLFilename == "data/maps/mapUS_environment.xml" and map.title ~= g_i18n:getText("mapUS_title") then
					g_logManager:info("Default lighting configuration: [%s] '%s'.", map.title, environmentXMLFilename)
				else
					local xmlFile = loadXMLFile("lightingXML", environmentXMLFilename)

					if xmlFile ~= nil then
						if hasXMLProperty(xmlFile, string.format("environment.envMap.timeProbe(%d)", 0)) then
							g_currentMission.baseDirectory = map.baseDirectory

							local lighting = Lighting:new(g_currentMission.environment)
							lighting:load(xmlFile, "environment")

							table.insert(customLighting, {lighting = lighting, name = map.title})
							table.insert(customLightingTexts, string.format("%s (%s)", map.title, g_i18n:getText("ui_map")))

							g_logManager:info("Lighting configuration loaded: [%s] '%s'.", map.title, environmentXMLFilename)
						else
							g_logManager:warning("Lighting configuration has no envMapTimes: [%s] '%s'.", map.title, environmentXMLFilename)
						end

						delete(xmlFile)
					end
				end
			end

			delete(mapXML)
		end
	end

	g_currentMission.baseDirectory = baseDirectoryBackup

	return customLighting, customLightingTexts
end

function LightingSettings:getLightingStateByName(name)
	for i, lighting in pairs(self.customLighting) do
		if lighting.name == name then
			return i
		end
	end

	return 0
end

function LightingSettings:consoleCommandReloadCustomLighting()
	self.state = 0
	self:onStateChange(0, nil, false)

	g_currentMission.xmlFile = loadXMLFileFromMemory("mapXML", "<map/>")
	self.customLighting, self.customLightingTexts = self:loadCustomLightingConfigurations(self.lightingDirectory)
	delete(g_currentMission.xmlFile)
	g_currentMission.xmlFile = nil

	local optionElement = g_additionalSettingsManager.getSettingElement(self)
	optionElement:setTexts(self.customLightingTexts)
	optionElement:setState(1)

	return string.format("Custom lighting settings updated, available configurations: %d", #self.customLighting)
end

function LightingSettings.onCloseCallback(setting, object, element)
	setting:onStateChange(setting.state, nil, false)
end

--
-- BrightnessSettings
--

BrightnessSettings = {}

function BrightnessSettings:new()
	AdditionalSettingsUtil.registerEventListener("onLoad", self)

	self.name = "brightness"
	self.state = tonumber(string.format("%.0f", (getBrightness() - 0.5) / 0.1))
	self.loadState = AdditionalSettingsManager.LOAD_STATE.MISSION_START

	return self
end

function BrightnessSettings:onLoad(filename)
	AdditionalSettingsUtil.appendedFunction(g_shopConfigScreen.elements[1], "onCloseCallback", self, "onCloseCallback")
end

function BrightnessSettings:onCreateElement(optionElement)
	local texts = {}

	for i = 50, 300 + 0.0001, 10 do
		table.insert(texts, string.format("%d%%", i))
	end

	optionElement:setTexts(texts)
end

function BrightnessSettings:getBrightnessFromIndex(index)
	return MathUtil.clamp(0.5 + 0.1 * index, 0.5, 3)
end

function BrightnessSettings:onStateChange(state, optionElement, loadFromSavegame)
	setSceneBrightness(self:getBrightnessFromIndex(state))
end

function BrightnessSettings.onCloseCallback(setting, object, element)
	setting:onStateChange(setting.state, nil, false)
end

--
-- DOFSettings
--

DOFSettings = {}

function DOFSettings:new()
	AdditionalSettingsUtil.registerEventListener("onMissionStarted", self)

	self.name = "dof"
	self.active = true
	self.loadState = AdditionalSettingsManager.LOAD_STATE.MISSION_START

	return self
end

--[[function DOFSettings:onCreateElement(checkboxElement)
	checkboxElement:setVisible(Utils.getPerformanceClassId() >= GS_PROFILE_HIGH)
end]]

function DOFSettings:onMissionStarted(filename)
	self.initialStateBackup = g_depthOfFieldManager.initialState
	self.blurStateBackup = g_depthOfFieldManager.blurState

	self.initialStateDisabled = {0, 0, 0, math.huge, math.huge}
	self.blurStateDisabled = {0, 0, 0, math.huge, math.huge}
end

function DOFSettings:onStateChange(state, checkboxElement, loadFromSavegame)
	local initialState = self.initialStateDisabled
	local blurState = self.blurStateDisabled

	if state then
		initialState = self.initialStateBackup
		blurState = self.blurStateBackup
	end

	g_depthOfFieldManager.initialState = initialState
	g_depthOfFieldManager.blurState = blurState

	g_depthOfFieldManager:reset()

	if g_depthOfFieldManager.blurIsActive then
		g_depthOfFieldManager.oldSetDoFparams(unpack(g_depthOfFieldManager.blurState))
	end
end

--
-- BirdsSettings
--

BirdsSettings = {}

function BirdsSettings:new()
	AdditionalSettingsUtil.overwrittenFunction(LightWildlife, "removeFarAwayAnimals", self, "removeFarAwayAnimals")

	self.name = "birds"
	self.active = true
	self.loadState = AdditionalSettingsManager.LOAD_STATE.LOAD_MAP

	return self
end

function BirdsSettings.removeFarAwayAnimals(setting, object, superFunc, maxDistance, refPosX, refPosY, refPosZ)
	if not setting.active then
		maxDistance = 0
	end

	return superFunc(object, maxDistance, refPosX, refPosY, refPosZ)
end

--
-- AnimalsCollisionsSettings
--

AnimalsCollisionsSettings = {}

function AnimalsCollisionsSettings:new()
	AdditionalSettingsUtil.overwrittenStaticFunction(getfenv(0), "createAnimalHusbandry", self, "createAnimalHusbandry")

	self.name = "animalsCollisions"
	self.active = true
	self.loadState = AdditionalSettingsManager.LOAD_STATE.LOAD_MAP

	self.restartRequired = false

	return self
end

function AnimalsCollisionsSettings:onCreateElement(checkboxElement)
	checkboxElement:setVisible(not g_currentMission.missionDynamicInfo.isMultiplayer)
end

function AnimalsCollisionsSettings:onStateChange(state, checkboxElement, loadFromSavegame)
	if self.restartRequired then
		if g_gui.currentGuiName == "InGameMenu" then
			g_gui:showInfoDialog({
				text = g_i18n:getText("additionalSettings_dialog_animalsCollisions"),
				dialogType = DialogElement.TYPE_INFO,
				target = checkboxElement,
				callback = self.onAcceptDialog
			})
		end
	end
end

function AnimalsCollisionsSettings.onAcceptDialog(checkboxElement)
	local textElement = checkboxElement.elements[4]
	local text = string.format("%s*", g_i18n:getText("additionalSettings_animalsCollisions"))

	if textElement.text ~= text then
		textElement:setText(text)
		checkboxElement.toolTipText = string.format("%s\n*%s", checkboxElement.toolTipText, g_i18n:getText("additionalSettings_dialog_animalsCollisions"))
	end
end

function AnimalsCollisionsSettings.createAnimalHusbandry(setting, superFunc, animalType, navMeshNode, animalHusbandryXMLFilename, placementRaycastDistance, collisionMaskFilter)
	setting.restartRequired = true

	if not setting.active and collisionMaskFilter == 4294967295 then
		collisionMaskFilter = 268435456
	end

	return superFunc(animalType, navMeshNode, animalHusbandryXMLFilename, placementRaycastDistance, collisionMaskFilter)
end

--
-- CameraCollisionsSettings
--

CameraCollisionsSettings = {}

function CameraCollisionsSettings:new()
	if not g_modIsLoaded["FS19_disableVehicleCameraCollision"] then
		AdditionalSettingsUtil.overwrittenFunction(VehicleCamera, "getCollisionDistance", self, "getCollisionDistance")
	end

	self.name = "cameraCollisions"
	self.active = true
	self.loadState = AdditionalSettingsManager.LOAD_STATE.LOAD_MAP

	return self
end

function CameraCollisionsSettings:onCreateElement(checkboxElement)
	checkboxElement:setDisabled(g_modIsLoaded["FS19_disableVehicleCameraCollision"])
end

function CameraCollisionsSettings.getCollisionDistance(setting, object, superFunc)
	if not setting.active then
		return false, nil, nil, nil, nil, nil
	end

	return superFunc(object)
end

--
-- GuiCameraSettings
--

GuiCameraSettings = {}

function GuiCameraSettings:new()
	AdditionalSettingsUtil.overwrittenFunction(g_currentMission.guiTopDownCamera, "getMouseEdgeScrollingMovement", self, "getMouseEdgeScrollingMovement")

	self.name = "guiCamera"
	self.active = true
	self.loadState = AdditionalSettingsManager.LOAD_STATE.LOAD_MAP

	return self
end

function GuiCameraSettings.getMouseEdgeScrollingMovement(setting, object, superFunc, posX, posY, isDown, isUp, button)
	if not setting.active then
		return 0, 0
	end

	return superFunc(object, posX, posY, isDown, isUp, button)
end