-- 	DynamicAttach Script
--
-- 	Author: 	Heady
-- 	Convert:	Silas770
-- 	Date:		12.02.2019
--	Version:	1.0.0.0 (FS19)
--

local modName = g_currentModName;

DynamicAttach = {};

function DynamicAttach.prerequisitesPresent(specializations)
    return true;
end;

function DynamicAttach.registerFunctions(vehicleType)
	SpecializationUtil.registerFunction(vehicleType, "findDynamicObjectCallback", DynamicAttach.findDynamicObjectCallback);
	SpecializationUtil.registerFunction(vehicleType, "setDynamicAttach", DynamicAttach.setDynamicAttach);
	SpecializationUtil.registerFunction(vehicleType, "getDynamicAttachSpecTable", DynamicAttach.getDynamicAttachSpecTable);
end;

function DynamicAttach.registerEventListeners(vehicleType)
	local els = {"onLoad", "onReadStream", "onWriteStream", "onUpdateTick", "onRegisterActionEvents"};
	for _, el in pairs(els) do
		SpecializationUtil.registerEventListener(vehicleType, el, DynamicAttach);
	end;
end;

function DynamicAttach:onLoad(savegame)
	local spec = self:getDynamicAttachSpecTable();
	
	spec.attached = false;
	spec.doStateChange = false;
	
	spec.objectsInRange = {};
	spec.objectsToJoint = {};
	
	spec.raycastNode = I3DUtil.indexToObject(self.components, getXMLString(self.xmlFile, "vehicle.dynamicAttach#raycastNode"), self.i3dMappings);
	spec.boxSizeX = getXMLFloat(self.xmlFile, "vehicle.dynamicAttach#boxSizeX", 1);
	spec.boxSizeY = getXMLFloat(self.xmlFile, "vehicle.dynamicAttach#boxSizeY", 1);
	spec.boxSizeZ = getXMLFloat(self.xmlFile, "vehicle.dynamicAttach#boxSizeZ", 1);
	
	if spec.raycastNode == nil then
		g_logManager:xmlWarning(self.configFileName, "'dynamicAttach#raycastNode' is not defined - DynamicAttach will not work")
	end;
end;

function DynamicAttach:onReadStream(streamId, connection)
	local spec = self:getDynamicAttachSpecTable();
	spec.attached = streamReadBool(streamId);
end;

function DynamicAttach:onWriteStream(streamId, connection)
	local spec = self:getDynamicAttachSpecTable();
	streamWriteBool(streamId, spec.attached);
end;

function DynamicAttach:onUpdateTick(dt)
	if self.isServer then
		local spec = self:getDynamicAttachSpecTable();

		if spec.doStateChange and spec.raycastNode ~= nil then
			spec.objectsInRange = {};
			local x,y,z = getWorldTranslation(spec.raycastNode);
			local rx,ry,rz = getWorldRotation(spec.raycastNode);
			
			local sizeX, sizeY, sizeZ = spec.boxSizeX, spec.boxSizeY, spec.boxSizeZ;
			overlapBox(x, y, z, rx, ry, rz, sizeX, sizeY, sizeZ, "findDynamicObjectCallback", self, nil, true, false, true);
			
			local actionEvent = spec.actionEvents[InputAction.DYNAMICATTACH_ATTACH];
			local actionEventText;
			
			if spec.attached then
				-- create joints
				for index, object in pairs(spec.objectsInRange) do
					if spec.objectsToJoint[index] == nil then
						local constr = JointConstructor:new();
						constr:setActors(self.components[1].node, object.physics);
						-- create joint at center of mass to avoid jittering (e.g. tree trunks pivots are not at center of mass position)
						local x,y,z = localToWorld(object.physics, getCenterOfMass(object.physics));
						constr:setJointWorldPositions(x,y,z, x,y,z);
						constr:setRotationLimit(0, 0, 0);
						constr:setRotationLimit(1, 0, 0);
						constr:setRotationLimit(2, 0, 0);
						object.dynamicAttacherJoint = constr:finalize();
						spec.objectsToJoint[index] = object;
					end;
				end;
				
				actionEventText = g_i18n:getText("DYNAMICATTACH_DETACH");
			else
				-- remove joints
				for index, object in pairs(spec.objectsToJoint) do
					if spec.objectsToJoint[index] ~= nil then
						removeJoint(object.dynamicAttacherJoint);
						spec.objectsToJoint[index] = nil;
					end;
				end;
				
				actionEventText = g_i18n:getText("DYNAMICATTACH_ATTACH");
			end;
				
			spec.doStateChange = false;
			g_inputBinding:setActionEventText(actionEvent.actionEventId, actionEventText);
		end;
	end;
end;

function DynamicAttach:findDynamicObjectCallback(transformId)
	if transformId ~= 0 and getHasClassId(transformId, ClassIds.SHAPE) then
		local spec = self:getDynamicAttachSpecTable();
		
        local object = g_currentMission:getNodeObject(transformId);
        if object ~= nil then
            if object.getMeshNodes ~= nil and object.dynamicMountObject == nil and spec.objectsInRange[transformId] == nil then
                local nodes = object:getMeshNodes();
                if nodes ~= nil then
                    spec.objectsInRange[transformId] = {physics=object.nodeId, visuals=nodes, object=object};
                end;
            end;
        elseif getSplitType(transformId) ~= 0 then
            local rigidBodyType = getRigidBodyType(transformId);
            if (rigidBodyType == "Dynamic" or rigidBodyType == "Kinematic") and spec.objectsInRange[transformId] == nil then
                spec.objectsInRange[transformId] = {physics=transformId, visuals={transformId}};
            end;
        end;
    end;
	
    return true;
end;

function DynamicAttach:setDynamicAttach(noEventSend)
	local spec = self:getDynamicAttachSpecTable();

	spec.attached = not spec.attached;
	spec.doStateChange = true;
	
	DynamicAttachEvent.sendEvent(self, spec.attached, noEventSend);
end;

function DynamicAttach:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
	if self.isClient then
		local spec = self:getDynamicAttachSpecTable();
		self:clearActionEventsTable(spec.actionEvents);
		
		if isActiveForInputIgnoreSelection then
			local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.DYNAMICATTACH_ATTACH, self, DynamicAttach.actionEventDynamicAttach, false, true, false, true, nil)
            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_HIGH);
			
			local actionEventText;
			if spec.attached then
				actionEventText = g_i18n:getText("DYNAMICATTACH_DETACH");
			else
				actionEventText = g_i18n:getText("DYNAMICATTACH_ATTACH");
			end;
			g_inputBinding:setActionEventText(actionEventId, actionEventText);
		end;
	end;
end;

function DynamicAttach.actionEventDynamicAttach(self, actionName, inputValue, callbackState, isAnalog)
	self:setDynamicAttach();
end;

function DynamicAttach:getDynamicAttachSpecTable()
	return self["spec_" .. modName .. ".DynamicAttach"];
end;


-- MP Event
DynamicAttachEvent = {};
DynamicAttachEvent_mt = Class(DynamicAttachEvent, Event);

InitEventClass(DynamicAttachEvent, "DynamicAttachEvent");

function DynamicAttachEvent:emptyNew()
    local self = Event:new(DynamicAttachEvent_mt);
    self.className = "DynamicAttachEvent";
    return self;
end;

function DynamicAttachEvent:new(object, attached)
    local self = DynamicAttachEvent:emptyNew()
    self.object = object;
	self.attached = attached;
    return self;
end;

function DynamicAttachEvent:readStream(streamId, connection)
	self.object = NetworkUtil.readNodeObject(streamId);
    self.attached = streamReadBool(streamId);
    self:run(connection);
end;

function DynamicAttachEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.object);
    streamWriteUInt8(streamId, self.attached);
end;

function DynamicAttachEvent:run(connection)
    if not connection:getIsServer() then
        g_server:broadcastEvent(self, nil, connection, self.object);
    end;
	if self.object ~= nil then
		self.object:setDynamicAttach(true);
	end;
end;

function DynamicAttachEvent.sendEvent(vehicle, attached, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(DynamicAttachEvent:new(vehicle, attached), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(DynamicAttachEvent:new(vehicle, attached));
		end;
	end;
end;