--[[
	Author: 	derelky
	Date:		18.07.2019
	Version:	1.0
	
	History:
				v1.0 @ 18.07.2019 - initial implementation in FS 19
]]
local modDirectory = g_currentModDirectory

MixerWagonconf = {};
MixerWagonconf.MOD_NAME = g_currentModName

source(modDirectory .. "Stade_ZW4010_FA/scripts/MixerWagonHUDExtension.lua")

function MixerWagonconf.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Trailer, specializations) and SpecializationUtil.hasSpecialization(TurnOnVehicle, specializations);
end

function MixerWagonconf.registerFunctions(vehicleType)
end

function MixerWagonconf.registerOverwrittenFunctions(vehicleType)
	SpecializationUtil.registerOverwrittenFunction(vehicleType, "addFillUnitFillLevel", MixerWagonconf.addFillUnitFillLevel)
	SpecializationUtil.registerOverwrittenFunction(vehicleType, "getFillUnitAllowsFillType", MixerWagonconf.getFillUnitAllowsFillType)
	SpecializationUtil.registerOverwrittenFunction(vehicleType, "getDischargeFillType", MixerWagonconf.getDischargeFillType)    
end

function MixerWagonconf.registerEventListeners(vehicleType)
	SpecializationUtil.registerEventListener(vehicleType, "onLoad", MixerWagonconf)
    --SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", MixerWagonconf)
	SpecializationUtil.registerEventListener(vehicleType, "onDelete", MixerWagonconf)
	SpecializationUtil.registerEventListener(vehicleType, "onReadStream", MixerWagonconf)
	SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", MixerWagonconf)
	SpecializationUtil.registerEventListener(vehicleType, "onReadUpdateStream", MixerWagonconf)
	SpecializationUtil.registerEventListener(vehicleType, "onWriteUpdateStream", MixerWagonconf)
	SpecializationUtil.registerEventListener(vehicleType, "onUpdate", MixerWagonconf)
	SpecializationUtil.registerEventListener(vehicleType, "onTurnedOn", MixerWagonconf)
	SpecializationUtil.registerEventListener(vehicleType, "onTurnedOff", MixerWagonconf)
	SpecializationUtil.registerEventListener(vehicleType, "onFillUnitFillLevelChanged", MixerWagonconf)
end

function MixerWagonconf.initSpecialization()
    g_configurationManager:addConfigurationType("mixerWagon", g_i18n:getText("configuration_mixerWagon"), "mixerWagon", nil, nil, nil, ConfigurationUtil.SELECTOR_MULTIOPTION)
end

function MixerWagonconf:onLoad(savegame)
	
	local spec = self.spec_MixerWagonconf
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, self.configFileName, "vehicle.mixerWagonPickupStartSound", "vehicle.turnOnVehicle.sounds.start")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, self.configFileName, "vehicle.mixerWagonPickupStopSound", "vehicle.turnOnVehicle.sounds.stop")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, self.configFileName, "vehicle.mixerWagonPickupSound", "vehicle.turnOnVehicle.sounds.work")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, self.configFileName, "vehicle.mixerWagonRotatingParts.mixerWagonRotatingPart#type", "vehicle.mixerWagon.mixAnimationNodes.animationNode", "mixerWagonMix")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, self.configFileName, "vehicle.mixerWagonRotatingParts.mixerWagonRotatingPart#type", "vehicle.mixerWagon.pickupAnimationNodes.animationNode", "mixerWagonPickup")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, self.configFileName, "vehicle.mixerWagonRotatingParts.mixerWagonScroller", "vehicle.mixerWagon.pickupAnimationNodes.pickupAnimationNode")
    
	local mixerWagonConfigurationId = Utils.getNoNil(self.configurations["mixerWagon"], 1)
    local baseKey = string.format("vehicle.mixerWagon.mixerWagonConfigurations.mixerWagonConfiguration(%d).mixerWagonFillTypes", mixerWagonConfigurationId -1)
    ObjectChangeUtil.updateObjectChanges(self.xmlFile, "vehicle.mixerWagon.mixerWagonConfigurations.mixerWagonConfiguration", mixerWagonConfigurationId , self.components, self)


	if self.isClient then
		spec.mixAnimationNodes = g_animationManager:loadAnimations(self.xmlFile, "vehicle.mixerWagon.mixAnimationNodes", self.components, self, self.i3dMappings)
		spec.pickupAnimationNodes = g_animationManager:loadAnimations(self.xmlFile, "vehicle.mixerWagon.pickupAnimationNodes", self.components, self, self.i3dMappings)
	end

	spec.activeTimerMax = 5000
	spec.activeTimer = 0
	spec.fillUnitIndex = Utils.getNoNil(getXMLInt(self.xmlFile, "vehicle.mixerWagon#fillUnitIndex"), 1)
	local fillUnit = self:getFillUnitByIndex(spec.fillUnitIndex)

	if fillUnit ~= nil then
		fillUnit.needsSaving = false

		if fillUnit.supportedFillTypes[FillType.GRASS_WINDROW] then
			fillUnit.supportedFillTypes[FillType.GRASS_WINDROW] = nil
		end
	end

	fillUnit.synchronizeFillLevel = false
	spec.mixerWagonFillTypes = {}
	spec.fillTypeToMixerWagonFillType = {}
	spec.outputfillType = g_fillTypeManager:getFillTypeIndexByName("FORAGE")
	local sumRatio = 0
    local i=0
    while true do
        local baseName = string.format("%s.mixerWagonFillType(%d)", baseKey, i) --string.format("vehicle.mixerWagon.mixerWagonFillTypes.mixerWagonFillType(%d)", i)
        local fillTypeStroutput = getXMLString(self.xmlFile,string.format("%s.mixerWagonoutput.fillType#fillType", baseKey)) --string.format("vehicle.mixerWagon.mixerWagonFillTypes.mixerWagonFillType(%d)", i)
        local fillTypeNotComplete = getXMLString(self.xmlFile,string.format("%s.mixerWagonoutput.fillType#fillTypeMixing", baseKey)) --string.format("vehicle.mixerWagon.mixerWagonFillTypes.mixerWagonFillType(%d)", i)
		if fillTypeStroutput~=nil then
			spec.outputfillType = g_fillTypeManager:getFillTypeIndexByName(fillTypeStroutput)
		else
			spec.outputfillType = g_fillTypeManager:getFillTypeIndexByName("FORAGE")
		end		
		if fillTypeNotComplete~=nil then
			spec.mixingfillType = g_fillTypeManager:getFillTypeIndexByName(fillTypeNotComplete)
		else
			spec.mixingfillType = g_fillTypeManager:getFillTypeIndexByName("FORAGE_MIXING")
		end
        if not hasXMLProperty(self.xmlFile, baseName) then
            break
        end
        local entry = {}
        entry.fillTypes = {}
        local j=0
        while true do
            local fillTypeKey = baseName..string.format(".fillType(%d)", j)
            if not hasXMLProperty(self.xmlFile, fillTypeKey) then
                break
            end
            local fillTypeStr = getXMLString(self.xmlFile, fillTypeKey.."#fillType")
            if fillTypeStr ~= nil then
                local fillTypeIndex = g_fillTypeManager:getFillTypeIndexByName(fillTypeStr)
                -- we do not want to support grass in mixer wagons becasue this would make hay "useless"
                if fillTypeIndex ~= nil and fillTypeIndex ~= FillType.GRASS_WINDROW then
                    if spec.fillTypeToMixerWagonFillType[fillTypeIndex] == nil then
                        entry.fillTypes[fillTypeIndex] = true
                        spec.fillTypeToMixerWagonFillType[fillTypeIndex] = entry
                    else
                        g_logManager:xmlWarning(self.configFileName, "MixerWagonFillType '%s' in '%s' already used! Ignoring it!", fillTypeKey, fillTypeStr)
                    end
                else
                    g_logManager:xmlWarning(self.configFileName, "FillType '%s' not defined for mixerWagonFillType '%s'!", fillTypeStr, fillTypeKey)
                end
            end
            j = j+1
        end
        entry.name = Utils.getNoNil(getXMLString(self.xmlFile, baseName.."#name"), "unknown")
        entry.minPercentage = Utils.getNoNil(getXMLFloat(self.xmlFile, baseName.."#minPercentage"), 0)*0.01
        entry.maxPercentage = Utils.getNoNil(getXMLFloat(self.xmlFile, baseName.."#maxPercentage"), 100)*0.01
        local ratiobackup = entry.maxPercentage  - entry.minPercentage
        entry.ratio = Utils.getNoNil(getXMLFloat(self.xmlFile, baseName.."#ratio"), ratiobackup)
        entry.fillLevel = 0
        if next(entry.fillTypes) ~= nil then
            sumRatio = sumRatio + entry.ratio
            table.insert(spec.mixerWagonFillTypes, entry)
        end
        i = i+1
    end
    for i, entry in ipairs(spec.mixerWagonFillTypes) do
        entry.ratio = entry.ratio / sumRatio
    end

	spec.dirtyFlag = self:getNextDirtyFlag()

	if savegame ~= nil then
		for i, entry in ipairs(spec.mixerWagonFillTypes) do
			local fillTypeKey = savegame.key .. string.format(".MixerWagonconf.fillType(%d)#fillLevel", i - 1)
			local fillLevel = Utils.getNoNil(getXMLFloat(savegame.xmlFile, fillTypeKey), 0)

			if fillLevel > 0 then
				self:addFillUnitFillLevel(self:getOwnerFarmId(), spec.fillUnitIndex, fillLevel, next(entry.fillTypes), ToolType.UNDEFINED, nil)
			end
		end
	end
end

function MixerWagonconf:addFillUnitFillLevel(superFunc, farmId, fillUnitIndex, fillLevelDelta, fillTypeIndex, toolType, fillPositionData)
	
    local spec = self.spec_MixerWagonconf
    if fillUnitIndex ~= spec.fillUnitIndex then
        return superFunc(self, farmId, fillUnitIndex, fillLevelDelta, fillTypeIndex, toolType, fillPositionData)
    end
    local oldFillLevel = self:getFillUnitFillLevel(fillUnitIndex)
    local mixerWagonFillType = spec.fillTypeToMixerWagonFillType[fillTypeIndex]
    -- allow to put forage back to mixer. Split it up into valid forage mixing ratios
    if fillTypeIndex == spec.outputfillType and fillLevelDelta > 0 then
        for _, entry in pairs(spec.mixerWagonFillTypes) do
            local delta = fillLevelDelta * entry.ratio
            self:addFillUnitFillLevel(farmId, fillUnitIndex, delta, next(entry.fillTypes), toolType, fillPositionData)
        end
        return fillLevelDelta
    end
    if mixerWagonFillType == nil then
        -- used for discharge
        if fillLevelDelta < 0 and oldFillLevel > 0 then
            -- remove values from all fill types such that the ratio doesn't change
            fillLevelDelta = math.max(fillLevelDelta, -oldFillLevel)
            local newFillLevel = 0
            for _, entry in pairs(spec.mixerWagonFillTypes) do
                local entryDelta = fillLevelDelta * (entry.fillLevel / oldFillLevel)
                entry.fillLevel = math.max(entry.fillLevel + entryDelta, 0)
                newFillLevel = newFillLevel + entry.fillLevel
            end
            if newFillLevel < 0.1 then
                for _, entry in pairs(spec.mixerWagonFillTypes) do
                    entry.fillLevel = 0
                end
                fillLevelDelta = -oldFillLevel
            end
            self:raiseDirtyFlags(spec.dirtyFlag)
            local ret = superFunc(self, farmId, fillUnitIndex, fillLevelDelta, fillTypeIndex, toolType, fillPositionData)
            return ret
        end
        return 0
    end
    local capacity = self:getFillUnitCapacity(fillUnitIndex)
    local free = capacity - oldFillLevel
    if fillLevelDelta > 0 then
        mixerWagonFillType.fillLevel = mixerWagonFillType.fillLevel + math.min(free, fillLevelDelta)
        spec.activeTimer = spec.activeTimerMax
    else
        mixerWagonFillType.fillLevel = math.max(0, mixerWagonFillType.fillLevel + fillLevelDelta)
    end
    local newFillLevel = 0
    for _, mixerWagonFillType in pairs(spec.mixerWagonFillTypes) do
        newFillLevel = newFillLevel + mixerWagonFillType.fillLevel
    end
    newFillLevel = MathUtil.clamp(newFillLevel, 0, self:getFillUnitCapacity(fillUnitIndex))
    local newFillType = FillType.UNKNOWN
    local isSingleFilled = false
    local isForageOk = false
    for _, mixerWagonFillType in pairs(spec.mixerWagonFillTypes) do
        if newFillLevel == mixerWagonFillType.fillLevel then
            isSingleFilled = true
            newFillType = next(mixerWagonFillType.fillTypes)
            break
        end
    end
    if not isSingleFilled then
        isForageOk = true
        for _, mixerWagonFillType in pairs(spec.mixerWagonFillTypes) do
            if mixerWagonFillType.fillLevel < mixerWagonFillType.minPercentage * newFillLevel - 0.01 or mixerWagonFillType.fillLevel > mixerWagonFillType.maxPercentage * newFillLevel + 0.01 then
                isForageOk = false
                break
            end
        end
    end
    if isForageOk then
        newFillType = spec.outputfillType
    elseif not isSingleFilled then
        newFillType = spec.mixingfillType
    end
    self:raiseDirtyFlags(spec.dirtyFlag)
    self:setFillUnitFillType(fillUnitIndex, newFillType)
    return FillUnit.addFillUnitFillLevel(self, farmId, fillUnitIndex, newFillLevel-oldFillLevel, newFillType, toolType, fillPositionData)
end

function MixerWagonconf:onDelete()
	local spec = self.spec_MixerWagonconf

	if self.isClient then
		g_animationManager:deleteAnimations(spec.mixAnimationNodes)
		g_animationManager:deleteAnimations(spec.pickupAnimationNodes)
	end
end

function MixerWagonconf:saveToXMLFile(xmlFile, key, usedModNames)
	local spec = self.spec_MixerWagonconf

	for i, fillType in ipairs(spec.mixerWagonFillTypes) do
		local fillTypeKey = string.format("%s.fillType(%d)", key, i - 1)

		setXMLFloat(xmlFile, fillTypeKey .. "#fillLevel", fillType.fillLevel)
	end
end

function MixerWagonconf:onReadStream(streamId, connection)
	local spec = self.spec_MixerWagonconf

	for _, entry in ipairs(spec.mixerWagonFillTypes) do
		local fillLevel = streamReadFloat32(streamId)

		if fillLevel > 0 then
			self:addFillUnitFillLevel(self:getOwnerFarmId(), spec.fillUnitIndex, fillLevel, next(entry.fillTypes), ToolType.UNDEFINED, nil)
		end
	end
end

function MixerWagonconf:onWriteStream(streamId, connection)
	local spec = self.spec_MixerWagonconf

	for _, entry in ipairs(spec.mixerWagonFillTypes) do
		streamWriteFloat32(streamId, entry.fillLevel)
	end
end

function MixerWagonconf:onReadUpdateStream(streamId, timestamp, connection)
	if connection:getIsServer() and streamReadBool(streamId) then
		local spec = self.spec_MixerWagonconf

		for _, entry in ipairs(spec.mixerWagonFillTypes) do
			local fillLevel = streamReadFloat32(streamId)
			local delta = fillLevel - entry.fillLevel

			if delta ~= 0 then
				self:addFillUnitFillLevel(self:getOwnerFarmId(), spec.fillUnitIndex, delta, next(entry.fillTypes), ToolType.UNDEFINED, nil)
			end
		end
	end
end

function MixerWagonconf:onWriteUpdateStream(streamId, connection, dirtyMask)
	if not connection:getIsServer() then
		local spec = self.spec_MixerWagonconf

		if streamWriteBool(streamId, bitAND(dirtyMask, spec.dirtyFlag) ~= 0) then
			for _, entry in ipairs(spec.mixerWagonFillTypes) do
				streamWriteFloat32(streamId, entry.fillLevel)
			end
		end
	end
end

function MixerWagonconf:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	local spec = self.spec_MixerWagonconf
	local tipState = self:getTipState()
	local isTurnedOn = self:getIsTurnedOn()
	local isDischarging = tipState == Trailer.TIPSTATE_OPENING or tipState == Trailer.TIPSTATE_OPEN

	if spec.activeTimer > 0 or isTurnedOn or isDischarging then
		spec.activeTimer = spec.activeTimer - dt

		g_animationManager:startAnimations(spec.mixAnimationNodes)
	else
		g_animationManager:stopAnimations(spec.mixAnimationNodes)
	end
end

function MixerWagonconf:getFillUnitAllowsFillType(superFunc, fillUnitIndex, fillTypeIndex)
	local spec = self.spec_MixerWagonconf

	if spec.fillUnitIndex == fillUnitIndex then
		local mixerWagonFillType = spec.fillTypeToMixerWagonFillType[fillTypeIndex]

		if mixerWagonFillType ~= nil then
			return true
		end
	end

	return superFunc(self, fillUnitIndex, fillTypeIndex)
end

function MixerWagonconf:getDischargeFillType(superFunc, dischargeNode)
	local spec = self.spec_MixerWagonconf
	local fillUnitIndex = dischargeNode.fillUnitIndex

	if fillUnitIndex == spec.fillUnitIndex then
		local currentFillType = self:getFillUnitFillType(fillUnitIndex)
		local fillLevel = self:getFillUnitFillLevel(fillUnitIndex)

		if currentFillType == spec.mixingfillType and fillLevel > 0 then
			for _, entry in pairs(spec.mixerWagonFillTypes) do
				if entry.fillLevel > 0 then
					currentFillType = next(entry.fillTypes)

					break
				end
			end
		end

		return currentFillType
	end

	return superFunc(self, dischargeNode)
end

function MixerWagonconf:onFillUnitFillLevelChanged(fillUnitIndex, fillLevelDelta, fillTypeIndex, toolType, fillPositionData, appliedDelta)
	local spec = self.spec_MixerWagonconf

	if spec.fillUnitIndex == fillUnitIndex then
		local fillLevel = self:getFillUnitFillLevel(fillUnitIndex)

		if fillLevel == 0 then
			for _, entry in pairs(spec.mixerWagonFillTypes) do
				entry.fillLevel = 0
			end
		end
	end
end

function MixerWagonconf:onTurnedOn()
	if self.isClient then
		local spec = self.spec_MixerWagonconf

		g_animationManager:startAnimations(spec.pickupAnimationNodes)
	end
end

function MixerWagonconf:onTurnedOff()
	if self.isClient then
		local spec = self.spec_MixerWagonconf

		g_animationManager:stopAnimations(spec.pickupAnimationNodes)
	end
end

function Vehicle:own_getSpecTable(name)
    local spec = self["spec_" .. MixerWagonconf.MOD_NAME .. "." .. name]
    if spec ~= nil then
        return spec
    end

    return self["spec_" .. name]
end