-- script for tuning Motorized vehicles
-- V.1
-- By Marv63 2020
-- credits to PeterAH for base structure

local _modDirectory = g_currentModDirectory
tuning = {}; 

function tuning.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Motorized, specializations)
end

function tuning.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", tuning)
--    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", tuning)
end

function Motorized:loadMotor(xmlFile, motorId)
    local key, motorId = ConfigurationUtil.getXMLConfigurationKey(xmlFile, motorId, "vehicle.motorized.motorConfigurations.motorConfiguration", "vehicle.motorized", "motor")
    local spec = self.spec_motorized
    local fallbackConfigKey = "vehicle.motorized.motorConfigurations.motorConfiguration(0)"
    local fallbackOldKey = "vehicle"
    spec.motorType = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#type", getXMLString, "vehicle", fallbackConfigKey, fallbackOldKey)
    spec.motorStartAnimation = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#startAnimationName", getXMLString, "vehicle", fallbackConfigKey, fallbackOldKey)
    spec.fuelCapacity = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".fuelCapacity", "", getXMLFloat, 500, fallbackConfigKey, fallbackOldKey)
    spec.consumerConfigurationIndex = ConfigurationUtil.getConfigurationValue(xmlFile, key, "#consumerConfigurationIndex", "", getXMLInt, 1, fallbackConfigKey, fallbackOldKey)
    local wheelKey, _ = ConfigurationUtil.getXMLConfigurationKey(xmlFile, self.configurations["wheel"], "vehicle.wheels.wheelConfigurations.wheelConfiguration", "vehicle.wheels", "wheels")
    ObjectChangeUtil.updateObjectChanges(xmlFile, "vehicle.motorized.motorConfigurations.motorConfiguration", motorId, self.components, self)
    local motorMinRpm = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#minRpm", getXMLFloat, 1000, fallbackConfigKey, fallbackOldKey)
    local motorMaxRpm = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#maxRpm", getXMLFloat, 1800, fallbackConfigKey, fallbackOldKey)
    local minSpeed = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#minSpeed", getXMLFloat, 1, fallbackConfigKey, fallbackOldKey)
    local maxForwardSpeed = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#maxForwardSpeed", getXMLFloat, nil, fallbackConfigKey, fallbackOldKey)
    local maxBackwardSpeed = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#maxBackwardSpeed", getXMLFloat, nil, fallbackConfigKey, fallbackOldKey)
    if maxForwardSpeed ~= nil then
        maxForwardSpeed = maxForwardSpeed/3.6
    end
    if maxBackwardSpeed ~= nil then
        maxBackwardSpeed = maxBackwardSpeed/3.6
    end
    local maxWheelSpeed = ConfigurationUtil.getConfigurationValue(xmlFile, wheelKey, ".wheels", "#maxForwardSpeed", getXMLFloat, nil, nil, "vehicle.wheels")
    if maxWheelSpeed ~= nil then
        maxForwardSpeed = maxWheelSpeed/3.6
    end
    local accelerationLimit = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#accelerationLimit", getXMLFloat, 2.0, fallbackConfigKey, fallbackOldKey) -- m/s^2
    local brakeForce = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#brakeForce", getXMLFloat, 10, fallbackConfigKey, fallbackOldKey)*2
    local lowBrakeForceScale = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#lowBrakeForceScale", getXMLFloat, 0.5, fallbackConfigKey, fallbackOldKey)
    local lowBrakeForceSpeedLimit = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#lowBrakeForceSpeedLimit", getXMLFloat, 1, fallbackConfigKey, fallbackOldKey)/3600
    
    local storeItem = g_storeManager:getItemByXMLFilename(self.configFileName)
    local tuneConfigId = Utils.getNoNil(self.configurations["tuning"], 1)
    local cMultiplier = 1
    
    if storeItem.configurations["tuning"] ~= nil then
        cMultiplier = Utils.getNoNil(storeItem.configurations["tuning"][tuneConfigId].configMultiplier, 1)
    end


    local torqueScale = tonumber(string.format("%." .. 4 .. "f", ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#torqueScale", getXMLFloat, 1, fallbackConfigKey, fallbackOldKey) * cMultiplier))    



    local ptoMotorRpmRatio = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#ptoMotorRpmRatio", getXMLFloat, 4, fallbackConfigKey, fallbackOldKey)
    local minForwardGearRatio = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".transmission", "#minForwardGearRatio", getXMLFloat, nil, fallbackConfigKey, fallbackOldKey)
    local maxForwardGearRatio = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".transmission", "#maxForwardGearRatio", getXMLFloat, nil, fallbackConfigKey, fallbackOldKey)
    local minBackwardGearRatio = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".transmission", "#minBackwardGearRatio", getXMLFloat, nil, fallbackConfigKey, fallbackOldKey)
    local maxBackwardGearRatio = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".transmission", "#maxBackwardGearRatio", getXMLFloat, nil, fallbackConfigKey, fallbackOldKey)
    local gearChangeTime = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".transmission", "#gearChangeTime", getXMLFloat, nil, fallbackConfigKey, fallbackOldKey)
    local autoGearChangeTime = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".transmission", "#autoGearChangeTime", getXMLFloat, nil, fallbackConfigKey, fallbackOldKey)
    local axleRatio = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".transmission", "#axleRatio", getXMLFloat, 1.0, fallbackConfigKey, fallbackOldKey)
    if maxForwardGearRatio == nil or minForwardGearRatio == nil then
        minForwardGearRatio = nil
        maxForwardGearRatio = nil
    else
        minForwardGearRatio = minForwardGearRatio * axleRatio
        maxForwardGearRatio = maxForwardGearRatio * axleRatio
    end
    if minBackwardGearRatio == nil or maxBackwardGearRatio == nil then
        minBackwardGearRatio = nil
        maxBackwardGearRatio = nil
    else
        minBackwardGearRatio = minBackwardGearRatio * axleRatio
        maxBackwardGearRatio = maxBackwardGearRatio * axleRatio
    end
    -- Read forward gear ratios
    local forwardGearRatios
    if minForwardGearRatio == nil then
        forwardGearRatios = self:loadGears(xmlFile, "forwardGear", key, motorId, fallbackConfigKey, fallbackOldKey, motorMaxRpm, axleRatio)
        if forwardGearRatios == nil then
            print("Warning: Missing forward gear ratios for motor in '"..self.configFileName.."'!")
            forwardGearRatios = {1}
        end
    end
    -- Read backward gear ratios
    local backwardGearRatios
    if minBackwardGearRatio == nil then
        backwardGearRatios = self:loadGears(xmlFile, "backwardGear", key, motorId, fallbackConfigKey, fallbackOldKey, motorMaxRpm, axleRatio)
        if backwardGearRatios == nil then
            print("Warning: Missing backward gear ratios for motor in '"..self.configFileName.."'!")
            backwardGearRatios = {1}
        end
    end
    --local maxTorque = 0
    local torqueCurve = AnimCurve:new(linearInterpolator1)
    local torqueI = 0
    local torqueBase = fallbackOldKey..".motor.torque" -- fallback to old motor setup
    if key ~= nil and hasXMLProperty(xmlFile, fallbackConfigKey..".motor.torque(0)") then -- using default motor configuration
        torqueBase = fallbackConfigKey..".motor.torque"
    end
    if key ~= nil and hasXMLProperty(xmlFile, key..".motor.torque(0)") then -- using selected motor configuration
        torqueBase = key..".motor.torque"
    end
    while true do
        local torqueKey = string.format(torqueBase.."(%d)", torqueI)
        local normRpm = getXMLFloat(xmlFile, torqueKey.."#normRpm")
        local rpm
        if normRpm == nil then
            rpm = getXMLFloat(xmlFile, torqueKey.."#rpm")
        else
            rpm = normRpm * motorMaxRpm
        end
        local torque = getXMLFloat(xmlFile, torqueKey.."#torque")
        if torque == nil or rpm == nil then
            break
        end
        torqueCurve:addKeyframe({torque*torqueScale, time = rpm})
        torqueI = torqueI +1
    end
    spec.motor = VehicleMotor:new(self, motorMinRpm, motorMaxRpm, maxForwardSpeed, maxBackwardSpeed, torqueCurve, brakeForce, forwardGearRatios, backwardGearRatios, minForwardGearRatio, maxForwardGearRatio, minBackwardGearRatio, maxBackwardGearRatio, ptoMotorRpmRatio, minSpeed)
    local rotInertia = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#rotInertia", getXMLFloat, spec.motor:getRotInertia(), fallbackConfigKey, fallbackOldKey)
    local dampingRateFullThrottle = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#dampingRateFullThrottle", getXMLFloat, spec.motor:getDampingRateFullThrottle(), fallbackConfigKey, fallbackOldKey)
    local dampingRateZeroThrottleClutchEngaged = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#dampingRateZeroThrottleClutchEngaged", getXMLFloat, spec.motor:getDampingRateZeroThrottleClutchEngaged(), fallbackConfigKey, fallbackOldKey)
    local dampingRateZeroThrottleClutchDisengaged = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#dampingRateZeroThrottleClutchDisengaged", getXMLFloat, spec.motor:getDampingRateZeroThrottleClutchDisengaged(), fallbackConfigKey, fallbackOldKey)
    spec.motor:setRotInertia(rotInertia)
    spec.motor:setDampingRateFullThrottle(dampingRateFullThrottle)
    spec.motor:setDampingRateZeroThrottleClutchEngaged(dampingRateZeroThrottleClutchEngaged)
    spec.motor:setDampingRateZeroThrottleClutchDisengaged(dampingRateZeroThrottleClutchDisengaged)
    spec.motor:setLowBrakeForce(lowBrakeForceScale, lowBrakeForceSpeedLimit)
    spec.motor:setAccelerationLimit(accelerationLimit)
    local motorRotationAccelerationLimit = ConfigurationUtil.getConfigurationValue(xmlFile, key, ".motor", "#rpmSpeedLimit", getXMLFloat, nil, fallbackConfigKey, fallbackOldKey) -- xml: rpm/s -> converted to rad/s^2
    if motorRotationAccelerationLimit ~= nil then
        motorRotationAccelerationLimit = motorRotationAccelerationLimit * math.pi/30
        spec.motor:setMotorRotationAccelerationLimit(motorRotationAccelerationLimit)
    end
    if gearChangeTime ~= nil then
        spec.motor:setGearChangeTime(gearChangeTime*1000)
    end
    if autoGearChangeTime ~= nil then
        spec.motor:setAutoGearChangeTime(autoGearChangeTime*1000)
    end
end

function Motorized:loadConsumerConfiguration(xmlFile, consumerIndex)
    local key = string.format("vehicle.motorized.consumerConfigurations.consumerConfiguration(%d)", consumerIndex-1)
    local spec = self.spec_motorized
    local fallbackConfigKey = "vehicle.motorized.consumers"
    local fallbackOldKey = nil
    spec.consumers = {}
    spec.consumersByFillTypeName = {}
    spec.consumersByFillType = {}
    if not hasXMLProperty(xmlFile, key) then
        return
    end
    local i = 0
    while true do
        local consumerKey = string.format(".consumer(%d)", i)
        if not hasXMLProperty(xmlFile, key..consumerKey) then
            break
        end
        local consumer = {}
        consumer.fillUnitIndex = ConfigurationUtil.getConfigurationValue(xmlFile, key, consumerKey, "#fillUnitIndex", getXMLInt, 1, fallbackConfigKey, fallbackOldKey)
        local fillTypeName = ConfigurationUtil.getConfigurationValue(xmlFile, key, consumerKey, "#fillType", getXMLString, "consumer", fallbackConfigKey, fallbackOldKey)
        consumer.fillType = g_fillTypeManager:getFillTypeIndexByName(fillTypeName)
        local fillUnit = self:getFillUnitByIndex(consumer.fillUnitIndex)
        if fillUnit ~= nil then
            --fill fillUnit on start
            fillUnit.startFillLevel = fillUnit.capacity
            fillUnit.startFillTypeIndex = consumer.fillType
        else
            g_logManager:xmlWarning(self.configFileName, "Unknown fillUnit '%d' for consumer '%s'", consumer.fillUnitIndex, key..consumerKey)
            break
        end
        local storeItem = g_storeManager:getItemByXMLFilename(self.configFileName)
        local tuneConfigId = Utils.getNoNil(self.configurations["tuning"], 1)
        local cMultiplier = 1
        
        if storeItem.configurations["tuning"] ~= nil then
            cMultiplier = Utils.getNoNil(storeItem.configurations["tuning"][tuneConfigId].configMultiplier, 1)
        end
        local usage = ConfigurationUtil.getConfigurationValue(xmlFile, key, consumerKey, "#usage", getXMLFloat, 1.0, fallbackConfigKey, fallbackOldKey) * cMultiplier
        consumer.permanentConsumption = ConfigurationUtil.getConfigurationValue(xmlFile, key, consumerKey, "#permanentConsumption", getXMLBool, true, fallbackConfigKey, fallbackOldKey)
        if consumer.permanentConsumption then
            consumer.usage = usage / (60*60*1000)  -- from l/h to l/ms
        else
            consumer.usage = usage
        end
        consumer.refillLitersPerSecond = ConfigurationUtil.getConfigurationValue(xmlFile, key, consumerKey, "#refillLitersPerSecond", getXMLFloat, 0, fallbackConfigKey, fallbackOldKey)
        consumer.refillCapacityPercentage = ConfigurationUtil.getConfigurationValue(xmlFile, key, consumerKey, "#refillCapacityPercentage", getXMLFloat, 0, fallbackConfigKey, fallbackOldKey)
        table.insert(spec.consumers, consumer)
        spec.consumersByFillTypeName[fillTypeName] = consumer
        spec.consumersByFillType[consumer.fillType] = consumer
        i = i + 1
    end
end

function tuning:onLoad(savegame)
	local storeItem = g_storeManager:getItemByXMLFilename(self.configFileName)
    if storeItem ~= nil and storeItem.configurations ~= nil then
        if storeItem.configurations["tuning"] ~= nil then
			if FS19_Tuning.tMotor.config.shopConfig.useSRepair then
                print(storeItem)
				local id = self.configurations["tuning"]
				local repairMultiplier = storeItem.configurations["tuning"][id].repairMultiplier
				self.spec_wearable.workMultiplier = self.spec_wearable.workMultiplier * (1 / repairMultiplier)
				self.spec_wearable.fieldMultiplier = self.spec_wearable.fieldMultiplier * (1 / repairMultiplier) 
			end
        end
    end
end



