--[[
Interface: 1.7.1.0 b10490

Copyright (C) GtX (Andy), 2018

Author: GtX | Andy
Date: 08.10.2018

History:
V 1.0.1.0 @ 07.09.2019 - Fixed error allowing a player to try and move to driver seat in a vehicle being operated by another player.
                       - Add support for different link nodes in global vehicles. Example: 'Claas Xerion 5000 Series' with rotating cab.
                       - Add support for hand 'poseId' to be added in Global Vehicles. [ narrowFingers (Default), wideFingers, flatFingers ].
V 1.1.0.0 @ 02.12.2021 - Optimisations and support for new train in Alpine DLC
V 1.1.0.1 @ 29.03-2021 - Correct missing variable causing error when using Head Tracking (Thanks @Cornadius for the bug report).

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No changes are to be made to this script without permission from GtX | Andy

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
An diesem Skript dürfen ohne Genehmigung von GtX | Andy keine Änderungen vorgenommen werden
]]


UniversalPassengerSpec = {}

UniversalPassengerSpec.EXIT = 1
UniversalPassengerSpec.ENTER_DRIVER = 2
UniversalPassengerSpec.MOVE_VEHICLE = 3

function UniversalPassengerSpec.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Enterable, specializations) and not SpecializationUtil.hasSpecialization(Rideable, specializations)
end

function UniversalPassengerSpec.initSpecialization()
    g_configurationManager:addConfigurationType("universalPassenger", "Universal Passenger", nil, nil, nil, nil, ConfigurationUtil.SELECTOR_MULTIOPTION)
end

function UniversalPassengerSpec.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onPreLoad", UniversalPassengerSpec)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", UniversalPassengerSpec)
    SpecializationUtil.registerEventListener(vehicleType, "onDelete", UniversalPassengerSpec)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", UniversalPassengerSpec)
    SpecializationUtil.registerEventListener(vehicleType, "onPostUpdate", UniversalPassengerSpec)

    SpecializationUtil.registerEventListener(vehicleType, "onAIEnd", UniversalPassengerSpec)
end

function UniversalPassengerSpec.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "loadUniversalPassengerVehicle", UniversalPassengerSpec.loadUniversalPassengerVehicle)
    SpecializationUtil.registerFunction(vehicleType, "loadCharacterSpec", UniversalPassengerSpec.loadCharacterSpec)
    SpecializationUtil.registerFunction(vehicleType, "getNextSeat", UniversalPassengerSpec.getNextSeat)
    SpecializationUtil.registerFunction(vehicleType, "getIsPassenger", UniversalPassengerSpec.getIsPassenger)
    SpecializationUtil.registerFunction(vehicleType, "getIsPassengerVehicle", UniversalPassengerSpec.getIsPassengerVehicle)
    SpecializationUtil.registerFunction(vehicleType, "getHasPassengers", UniversalPassengerSpec.getHasPassengers)
    SpecializationUtil.registerFunction(vehicleType, "setPassengerCameraIndex", UniversalPassengerSpec.setPassengerCameraIndex)
    SpecializationUtil.registerFunction(vehicleType, "getNumPassangers", UniversalPassengerSpec.getNumPassangers)
    SpecializationUtil.registerFunction(vehicleType, "getIsSeatAvailable", UniversalPassengerSpec.getIsSeatAvailable)
    SpecializationUtil.registerFunction(vehicleType, "getAvailableSeats", UniversalPassengerSpec.getAvailableSeats)
    SpecializationUtil.registerFunction(vehicleType, "updateNumPassangers", UniversalPassengerSpec.updateNumPassangers)
    SpecializationUtil.registerFunction(vehicleType, "enterPassengerSeat", UniversalPassengerSpec.enterPassengerSeat)
    SpecializationUtil.registerFunction(vehicleType, "cyclePassengerSeats", UniversalPassengerSpec.cyclePassengerSeats)
    SpecializationUtil.registerFunction(vehicleType, "exitPassengerSeat", UniversalPassengerSpec.exitPassengerSeat)
    SpecializationUtil.registerFunction(vehicleType, "updateSeat", UniversalPassengerSpec.updateSeat)
    SpecializationUtil.registerFunction(vehicleType, "setPassengerCharacter", UniversalPassengerSpec.setPassengerCharacter)
    SpecializationUtil.registerFunction(vehicleType, "doSteeringWheelUpdate", UniversalPassengerSpec.doSteeringWheelUpdate)
    SpecializationUtil.registerFunction(vehicleType, "setPassengerMirrorVisible", UniversalPassengerSpec.setPassengerMirrorVisible)
    SpecializationUtil.registerFunction(vehicleType, "setPassengerAccessState", UniversalPassengerSpec.setPassengerAccessState)
    SpecializationUtil.registerFunction(vehicleType, "setConnectionToSeatId", UniversalPassengerSpec.setConnectionToSeatId)
end

function UniversalPassengerSpec.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "drawUIInfo", UniversalPassengerSpec.drawUIInfo)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getAllowCharacterVisibilityUpdate", UniversalPassengerSpec.getAllowCharacterVisibilityUpdate)

    SpecializationUtil.registerOverwrittenFunction(vehicleType, "playAnimation", UniversalPassengerSpec.playAnimation)
end

function UniversalPassengerSpec:onPreLoad(savegame)
    if g_universalPassenger == nil then
        self.spec_universalPassenger = {isPassengerVehicle = false}
        g_logManager:error("[UniversalPassengerSpec] - Something went wrong, 'UniversalPassenger' is missing!")

        return
    end

    self.spec_universalPassenger = self[string.format("spec_%s.universalPassenger", g_universalPassenger.customEnvironment)] or {}
    local spec = self.spec_universalPassenger

    spec.passengerSeats = {}

    spec.numPassengers = 0
    spec.numPassengerSeats = 0
    spec.availablePassengerSeats = 0

    spec.canMoveToDriverSeat = false
    spec.normalCamMode = true

    spec.actionEventsTable = {}
    spec.connectionToSeatId = {}

    spec.canExitSeat = false
    spec.canChangeSeats = false

    spec.isPassengerVehicle = false
    spec.farmAccessOnly = false

    spec.alpineDLCLoaded = g_modIsLoaded["pdlc_alpineFarmingPack"]
    spec.configurationId = self.configurations["universalPassenger"]

    if spec.configurationId ~= nil then
        local storeItem = g_storeManager:getItemByXMLFilename(self.configFileName)

        if storeItem ~= nil then
            local configuration = storeItem.configurations["universalPassenger"][spec.configurationId]

            if configuration ~= nil then
                spec.isPassengerVehicle = (configuration.index < 3)
                spec.farmAccessOnly = (configuration.index == 2)
            end
        end
    end
end

function UniversalPassengerSpec:onLoad(savegame)
    if g_universalPassenger == nil or not self:getIsPassengerVehicle() then
        return
    end

    local spec = self.spec_universalPassenger

    if hasXMLProperty(self.xmlFile, "vehicle.universalPassenger")then
        if spec.configurationId ~= nil then
            ObjectChangeUtil.updateObjectChanges(self.xmlFile, "vehicle.universalPassenger.accessConfiguration", spec.configurationId, self.components, self)
        end

        self:loadUniversalPassengerVehicle(self.xmlFile, "vehicle.universalPassenger", self.configFileName)

        spec.isGlobalVehicle = false
    else
        spec.isGlobalVehicle = true

        local globalVehicle = g_universalPassenger:getGlobalVehicleData(self.configFileName)

        if globalVehicle ~= nil then
            local xmlFile = loadXMLFile("universalPassengerXML", globalVehicle.xmlPath)
            local key = globalVehicle.key

            local xmlFilename = getXMLString(xmlFile, key .. "#xmlFilename")

            if xmlFilename ~= nil then
                if self.customEnvironment ~= nil then
                    local modName = getXMLString(xmlFile, key .. "#modName")

                    if modName ~= nil then
                        local directory = Utils.getNoNil(globalVehicle.directory, g_modsDirectory)
                        xmlFilename = string.format("%s%s/%s", directory, modName, xmlFilename)
                    end
                end

                if self.configFileName == xmlFilename then
                    self:loadUniversalPassengerVehicle(xmlFile, key, globalVehicle.xmlPath)
                else
                    local i = 0
                    while true do
                        key = string.format("universalPassengerVehicles.vehicle(%d)", i)
                        if not hasXMLProperty(xmlFile, key) then
                            break
                        end

                        xmlFilename = getXMLString(xmlFile, key .. "#xmlFilename")

                        if xmlFilename ~= nil then
                            if self.customEnvironment ~= nil then
                                local modName = getXMLString(xmlFile, key .. "#modName")

                                if modName ~= nil then
                                    local directory = Utils.getNoNil(globalVehicle.directory, g_modsDirectory)
                                    xmlFilename = string.format("%s%s/%s", directory, modName, xmlFilename)
                                end
                            end

                            if self.configFileName == xmlFilename then
                                self:loadUniversalPassengerVehicle(xmlFile, key, globalVehicle.xmlPath)
                                break
                            end
                        end

                        i = i + 1
                    end
                end
            end

            delete(xmlFile)
        end
    end

    spec.numPassengerSeats = #spec.passengerSeats
    spec.availablePassengerSeats = spec.numPassengerSeats

    spec.isPassengerVehicle = spec.availablePassengerSeats > 0

    if spec.isPassengerVehicle then
        g_universalPassenger:addPassengerVehicle(self)
    end
end

function UniversalPassengerSpec:loadUniversalPassengerVehicle(xmlFile, baseKey, xmlPath)
    local spec = self.spec_universalPassenger

    local usedAnimationNames = {}
    local modDir = g_universalPassenger.baseDirectory

    local i = 0
    while true do
        local seatKey = string.format("%s.passenger(%d)", baseKey, i)
        if not hasXMLProperty(xmlFile, seatKey)then
            break
        end

        local seat = {}
        local allowInsert = true

        local i3dCameraNodes
        local i3dCharacterNode
        local i3dCharacterNodeIndex = "0>"

        if spec.isGlobalVehicle then
            local i3dNode = g_i3DManager:loadSharedI3DFile(g_universalPassenger.baseI3D, modDir, false, false, false)

            if i3dNode ~= nil and i3dNode ~= 0 then
                i3dCameraNodes = {}

                i3dCharacterNode = I3DUtil.indexToObject(i3dNode, "0|0")
                i3dCameraNodes.outside = I3DUtil.indexToObject(i3dNode, "0|1")
                i3dCameraNodes.inside = I3DUtil.indexToObject(i3dNode, "0|2")

                if i3dCharacterNode ~= nil and i3dCameraNodes.outside ~= nil and i3dCameraNodes.inside ~= nil then
                    -- The character, inside camera and outside camera positions will be converted from 'component1' positioning to given link nodes if 'true'
                    -- This is used for redundancy if the given link node is not found due to a mod update, this will mean the player should still be sitting correctly.
                    local positionsFromComponent1 = Utils.getNoNil(getXMLBool(xmlFile, seatKey .. "#positionsFromComponent1"), false)

                    local characterLinkNodeStr = getXMLString(xmlFile, seatKey .. "#characterLinkNode")
                    local outsideCameraLinkNodeStr = getXMLString(xmlFile, seatKey .. "#outsideCameraLinkNode")
                    local insideCameraLinkNodeStr = getXMLString(xmlFile, seatKey .. "#insideCameraLinkNode")

                    local characterLinkNode, index1 = UniversalPassengerSpec.getLinkNodeToUse(self, characterLinkNodeStr)

                    if index1 ~= "0>" and positionsFromComponent1 then
                        seat.characterLinkNode = characterLinkNode
                    end

                    i3dCharacterNodeIndex = index1
                    link(characterLinkNode, i3dCharacterNode)

                    local outsideCameraLinkNode, index2 = UniversalPassengerSpec.getLinkNodeToUse(self, outsideCameraLinkNodeStr)

                    if index2 ~= "0>" and positionsFromComponent1 then
                        seat.outsideCameraLinkNode = outsideCameraLinkNode
                    end

                    link(outsideCameraLinkNode, i3dCameraNodes.outside)

                    local insideCameraLinkNode, index3 = UniversalPassengerSpec.getLinkNodeToUse(self, insideCameraLinkNodeStr)

                    if index3 ~= "0>" and positionsFromComponent1 then
                        seat.insideCameraLinkNode = insideCameraLinkNode
                    end

                    link(insideCameraLinkNode, i3dCameraNodes.inside)

                    if self.isServer then
                        seat.creatorInfo = {
                            characterLinkNode = characterLinkNode,
                            outsideCameraLinkNode = outsideCameraLinkNode,
                            insideCameraLinkNode = insideCameraLinkNode,
                            characterLinkNodeStr = characterLinkNodeStr,
                            outsideCameraLinkNodeStr = outsideCameraLinkNodeStr,
                            insideCameraLinkNodeStr = insideCameraLinkNodeStr
                        }
                    end
                else
                    allowInsert = false
                    g_universalPassenger:logPrint(UniversalPassenger.LOG_DEBUG, "Failed to load one or more required components from passenger i3d at %s!", modDir)
                end

                delete(i3dNode)
            else
                allowInsert = false
                g_universalPassenger:logPrint(UniversalPassenger.LOG_DEBUG, "No passenger i3d found at %s!", modDir)
            end

            local exitNodePositionStr = getXMLString(xmlFile, seatKey .. "#exitNodePosition")
            local exitNodePosition = StringUtil.getVectorNFromString(exitNodePositionStr, 3)

            if exitNodePosition ~= nil then
                seat.passengerExitNode = createTransformGroup("passengerExitNode")
                link(self.components[1].node, seat.passengerExitNode)
                setTranslation(seat.passengerExitNode, unpack(exitNodePosition))

                if self.isServer then
                    if seat.creatorInfo == nil then
                        seat.creatorInfo = {}
                    end

                    seat.creatorInfo.exitNodePositionStr = exitNodePositionStr
                end
            end
        else
            local exitNode = I3DUtil.indexToObject(self.components, getXMLString(xmlFile, seatKey .. "#exitNode"), self.i3dMappings)

            if exitNode ~= nil then
                seat.passengerExitNode = exitNode
            end
        end

        if allowInsert then
            -- Each animation can only be used on a single seat. These are played in the same way a standard 'enterAnimation' is played when entering.
            local enterAnimationName = getXMLString(xmlFile, seatKey .. ".enterAnimation#name")

            if enterAnimationName ~= nil then
                if self.getAnimationExists ~= nil and self:getAnimationExists(enterAnimationName) then
                    if usedAnimationNames[enterAnimationName] == nil then
                        usedAnimationNames[enterAnimationName] = true
                        seat.passengerEnterAnimation = enterAnimationName
                    else
                        g_universalPassenger:logPrint(UniversalPassenger.LOG_WARNING, "Animation '%s' already in use on another seat! (%s)", enterAnimationName, xmlPath)
                    end
                else
                    g_universalPassenger:logPrint(UniversalPassenger.LOG_INFO, "Animation '%s' is not available or specialization 'AnimatedVehicle' could not be found! (%s)", enterAnimationName, xmlPath)
                end
            end

            -- Parts need to already be in the vehicle and can not be imported for global vehicles.
            -- 'Max' defines when a seat is in use. If no 'Min' value is given the default position is used.
            if hasXMLProperty(xmlFile, seatKey .. ".partChange") then
                local indexString = getXMLString(xmlFile, seatKey .. ".partChange#index")
                local validName = getXMLString(xmlFile, seatKey .. ".partChange#validName")

                if validName == "" then
                    validName = nil
                end

                local index, isValid = UniversalPassengerSpec.getValidNodeFromIndexString(self.components, indexString, self.i3dMappings, validName)

                if index ~= nil then
                    if isValid and getRigidBodyType(index) ~= "Static" then
                        seat.partChange = {
                            index = index,
                            validName = validName or getName(index),
                            indexString = indexString
                        }

                        seat.partChange.transMin = Utils.getNoNil(StringUtil.getVectorNFromString(getXMLString(xmlFile, seatKey .. ".partChange#transMin"), 3), {getTranslation(index)})
                        seat.partChange.transMax = Utils.getNoNil(StringUtil.getVectorNFromString(getXMLString(xmlFile, seatKey .. ".partChange#transMax"), 3), {getTranslation(index)})

                        seat.partChange.rotMin = Utils.getNoNil(StringUtil.getRadiansFromString(getXMLString(xmlFile, seatKey .. ".partChange#rotMin"), 3), {getRotation(index)})
                        seat.partChange.rotMax = Utils.getNoNil(StringUtil.getRadiansFromString(getXMLString(xmlFile, seatKey .. ".partChange#rotMax"), 3), {getRotation(index)})

                        seat.partChange.scaleMin = Utils.getNoNil(StringUtil.getVectorNFromString(getXMLString(xmlFile, seatKey .. ".partChange#scaleMin"), 3), {getScale(index)})
                        seat.partChange.scaleMax = Utils.getNoNil(StringUtil.getVectorNFromString(getXMLString(xmlFile, seatKey .. ".partChange#scaleMax"), 3), {getScale(index)})

                        seat.partChange.visibilityMin = Utils.getNoNil(getXMLBool(xmlFile, seatKey .. ".partChange#visibilityMin"), true)
                        seat.partChange.visibilityMax = Utils.getNoNil(getXMLBool(xmlFile, seatKey .. ".partChange#visibilityMax"), true)

                        setRotation(index, unpack(seat.partChange.rotMin))
                        setScale(index, unpack(seat.partChange.scaleMin))
                        setVisibility(index, seat.partChange.visibilityMin)
                    else
                        if isValid then
                            g_universalPassenger:logPrint(UniversalPassenger.LOG_WARNING, "Rigid Body Type 'Static' is not supported for 'partChange' and will be ignored. (%s)", indexString, xmlPath)
                        else
                            g_universalPassenger:logPrint(UniversalPassenger.LOG_INFO, "Valid node could not be found for 'partChange' and will be ignored. (%s)", xmlPath)
                        end
                    end
                end
            end

            seat.cameras = {} -- Global loaded vehicles can have a max of 2 cameras. 1 x outside view, 1 x inside view.

            local j = 0
            while true do
                local cameraKey = string.format("%s.cameras.camera(%d)", seatKey, j)
                if not hasXMLProperty(xmlFile, cameraKey) or (spec.isGlobalVehicle and #seat.cameras > 2) then
                    break
                end

                local passengerCamera = VehicleCamera:new(self)
                if passengerCamera ~= nil then
                    local cameraLoaded = false

                    if spec.isGlobalVehicle then
                        cameraLoaded = UniversalPassengerSpec:loadCamearaFromXML(passengerCamera, i3dCameraNodes, j + 1, xmlFile, cameraKey, baseKey, seat)
                    else
                        cameraLoaded = passengerCamera:loadFromXML(xmlFile, cameraKey)
                    end

                    if cameraLoaded then
                        table.insert(seat.cameras, passengerCamera)
                    end
                end

                j = j + 1
            end

            seat.numCameras = #seat.cameras

            if seat.numCameras > 0 then
                seat.cameraIndex = 1
                spec.activeCamera = seat.cameras[1]

                if spec.isGlobalVehicle and seat.numCameras ~= 2 then
                    g_universalPassenger:logPrint(UniversalPassenger.LOG_ERROR, "Number of cameras found at '%s' does not equal two. Global vehicles need 2 x cameras. (%s)", seatKey, xmlPath)
                    g_universalPassenger:logPrint(UniversalPassenger.LOG_INFO, "camera1 should match the standard vehicle camera1. camera2 is a first person view camera.")
                    allowInsert = false
                end
            else
                g_universalPassenger:logPrint(UniversalPassenger.LOG_ERROR, "No cameras found for seat '%d' in xml file: %s", i, self.configFileName)
                allowInsert = false
            end

            if allowInsert then
                seat.used = false
                seat.passengerName = ""

                if i3dCharacterNode ~= nil then
                    local position = StringUtil.getVectorNFromString(getXMLString(xmlFile, seatKey .. ".passengerNode#position"), 3)
                    local rotation = Utils.getNoNil(StringUtil.getRadiansFromString(getXMLString(xmlFile, seatKey .. ".passengerNode#rotation"), 3), {0, 0, 0})

                    if seat.characterLinkNode then
                        position = {localToLocal(self.components[1].node, seat.characterLinkNode, position[1], position[2], position[3])}
                    end

                    setTranslation(i3dCharacterNode, unpack(position))
                    setRotation(i3dCharacterNode, unpack(rotation))
                else
                    local pIndex = I3DUtil.indexToObject(self.components, getXMLString(xmlFile, seatKey .. ".passengerNode#node"), self.i3dMappings)
                    i3dCharacterNode = getParent(pIndex)
                end

                if self.isServer then
                    if seat.creatorInfo == nil then
                        seat.creatorInfo = {}
                    end

                    seat.creatorInfo.i3dCharacterNode = i3dCharacterNode
                end

                seat.passengerCharacter = self:loadCharacterSpec(i3dCharacterNode, i3dCharacterNodeIndex, xmlFile, seatKey .. ".passengerNode")

                if seat.passengerCharacter ~= nil then
                    if seat.passengerCharacter.characterDistanceRefNode ~= nil then
                        seat.passengerRenderNode = seat.passengerCharacter.characterDistanceRefNode
                        seat.passengerRenderNodeOffset = {0, 1.5, 0}
                    else
                        seat.passengerRenderNode = self.components[1].node
                    end
                else
                    allowInsert = false
                end

                if allowInsert then
                    seat.id = #spec.passengerSeats + 1
                    table.insert(spec.passengerSeats, seat)
                end
            end
        end

        i = i + 1
    end
end

function UniversalPassengerSpec:loadCamearaFromXML(instance, i3dCameraNodes, id, xmlFile, key, baseKey, seat)
    -- This is an edit of [VehicleCamera:loadFromXML()] because I am loading my own cameras for global i3d not from XML.

    if id == 1 then
        instance.rotateNode = getChildAt(i3dCameraNodes.outside, 0)
        instance.cameraNode = getChildAt(instance.rotateNode, 0)

        local position = StringUtil.getVectorNFromString(getXMLString(xmlFile, key .. "#rotateNodePosition"), 3)

        if position ~= nil then
            if seat.outsideCameraLinkNode then
                position = {localToLocal(instance.vehicle.components[1].node, seat.outsideCameraLinkNode, position[1], position[2], position[3])}
            end

            setTranslation(instance.rotateNode, unpack(position))
        end

        local rotation = StringUtil.getRadiansFromString(getXMLString(xmlFile, key .. "#rotateNodeRotation"), 3)

        if rotation ~= nil then
            setRotation(instance.rotateNode, unpack(rotation))
        end
    else
        instance.cameraNode = i3dCameraNodes.inside
        --instance.shadowFocusBoxNode = i3dCameraNodes.shadowFocusBox

        -- Disabled by default until more info is found about this.
        if Utils.getNoNil(getXMLBool(xmlFile, baseKey .. "#allowShadowFocusBox"), false) then
            local shadowBoxIndexStr = getXMLString(instance.vehicle.xmlFile, "vehicle.enterable.cameras.camera(1)#shadowFocusBox")

            local shadowFocusBoxNode = I3DUtil.indexToObject(instance.vehicle.components, shadowBoxIndexStr, instance.vehicle.i3dMappings)

            if shadowFocusBoxNode ~= nil then
                if getHasClassId(shadowFocusBoxNode, ClassIds.SHAPE) then
                    instance.shadowFocusBoxNode = shadowFocusBoxNode
                else
                    g_universalPassenger:logPrint(UniversalPassenger.LOG_WARNING, "Invalid passenger camera shadow focus box '%s'. Must be a shape and cpu mesh", getName(shadowFocusBoxNode))
                end
            end
        end
    end

    if instance.cameraNode ~= nil then
        local cameraPosition = Utils.getNoNil(StringUtil.getVectorNFromString(getXMLString(xmlFile, key .. "#cameraPosition"), 3), {0, 0, 0})

        if seat.insideCameraLinkNode then
            cameraPosition = {localToLocal(instance.vehicle.components[1].node, seat.insideCameraLinkNode, cameraPosition[1], cameraPosition[2], cameraPosition[3])}
        end

        local cameraRotation = Utils.getNoNil(StringUtil.getRadiansFromString(getXMLString(xmlFile, key .. "#cameraRotation"), 3), {0, 0, 0})

        setTranslation(instance.cameraNode, unpack(cameraPosition))
        setRotation(instance.cameraNode, unpack(cameraRotation))

        instance.fovY = calculateFovY(instance.cameraNode)
        setFovY(instance.cameraNode, instance.fovY)
        instance.isRotatable = Utils.getNoNil(getXMLBool(xmlFile, key .. "#rotatable"), true)

        instance.limit = Utils.getNoNil(getXMLBool(xmlFile, key .. "#limit"), false)

        if instance.limit then
            instance.rotMinX = getXMLFloat(xmlFile, key .. "#rotMinX")
            instance.rotMaxX = getXMLFloat(xmlFile, key .. "#rotMaxX")
            instance.transMin = getXMLFloat(xmlFile, key .. "#transMin")
            instance.transMax = getXMLFloat(xmlFile, key .. "#transMax")

            if instance.rotMinX == nil or instance.rotMaxX == nil or instance.transMin == nil or instance.transMax == nil then
                g_universalPassenger:logPrint(UniversalPassenger.LOG_WARNING, "Missing 'rotMinX', 'rotMaxX', 'transMin' or 'transMax' for passenger camera '%s'", key)
                return false
            end
        end

        instance.isInside = Utils.getNoNil(getXMLBool(xmlFile, key .. "#isInside"), false)

        if instance.isInside then
            instance.defaultLowPassGain = Utils.getNoNil(getXMLFloat(xmlFile, key .. "#defaultLowPassGain"), 0.5)
            instance.defaultVolume = Utils.getNoNil(getXMLFloat(xmlFile, key .. "#defaultVolume"), 0.9)
        else
            instance.defaultLowPassGain = Utils.getNoNil(getXMLFloat(xmlFile, key .. "#defaultLowPassGain"), 1.0)
            instance.defaultVolume = Utils.getNoNil(getXMLFloat(xmlFile, key .. "#defaultVolume"), 1.0)
        end

        instance.allowHeadTracking = Utils.getNoNil(getXMLBool(xmlFile, key .. "#allowHeadTracking"), instance.isInside)
        instance.useOutdoorSounds = Utils.getNoNil(getXMLBool(xmlFile, key .. "#useOutdoorSounds"), not instance.isInside)

        if instance.isRotatable then
            instance.hasExtraRotationNode = instance.rotateNode ~= nil
        end

        local rotation = StringUtil.getRadiansFromString(getXMLString(xmlFile, key .. "#rotation"), 3)

        if rotation ~= nil then
            local rotationNode = instance.cameraNode

            if instance.rotateNode ~= nil then
                rotationNode = instance.rotateNode
            end

            setRotation(rotationNode, unpack(rotation))
        end

        local translation = StringUtil.getVectorNFromString(getXMLString(xmlFile, key.."#translation"), 3)

        if translation ~= nil then
            setTranslation(instance.cameraNode, unpack(translation))
        end

        instance.allowTranslation = (instance.rotateNode ~= nil and instance.rotateNode ~= instance.cameraNode)
        instance.useMirror = Utils.getNoNil(getXMLBool(xmlFile, key .. "#useMirror"), false)

        instance.useWorldXZRotation = getXMLBool(xmlFile, key .. "#useWorldXZRotation")
        instance.resetCameraOnVehicleSwitch = getXMLBool(xmlFile, key .. "#resetCameraOnVehicleSwitch")

        instance.positionSmoothingParameter = 0
        instance.lookAtSmoothingParameter = 0

        local useDefaultPositionSmoothing = Utils.getNoNil(getXMLBool(xmlFile, key .. "#useDefaultPositionSmoothing"), false)

        if useDefaultPositionSmoothing then
            if instance.isInside then
                instance.positionSmoothingParameter = 0.128
                instance.lookAtSmoothingParameter = 0.176
            else
                instance.positionSmoothingParameter = 0.016
                instance.lookAtSmoothingParameter = 0.022
            end
        end

        -- Do not allow change on server in case it is laggy, the camera moves to much.
        if not g_currentMission.missionDynamicInfo.isMultiplayer then
            instance.positionSmoothingParameter = Utils.getNoNil(getXMLFloat(xmlFile, key .. "#positionSmoothingParameter"), instance.positionSmoothingParameter)
            instance.lookAtSmoothingParameter = Utils.getNoNil(getXMLFloat(xmlFile, key .. "#lookAtSmoothingParameter"), instance.lookAtSmoothingParameter)
        end

        local useHeadTracking = g_gameSettings:getValue("isHeadTrackingEnabled") and isHeadTrackingAvailable() and instance.allowHeadTracking

        if useHeadTracking then
            instance.positionSmoothingParameter = 0
            instance.lookAtSmoothingParameter = 0
        end

        instance.cameraPositionNode = instance.cameraNode

        if instance.positionSmoothingParameter > 0 then
            instance.cameraPositionNode = createTransformGroup("cameraPositionNode")

            local camIndex = getChildIndex(instance.cameraNode)
            link(getParent(instance.cameraNode), instance.cameraPositionNode, camIndex)

            local x,y,z = getTranslation(instance.cameraNode)
            local rx,ry,rz = getRotation(instance.cameraNode)

            setTranslation(instance.cameraPositionNode, x,y,z)
            setRotation(instance.cameraPositionNode, rx,ry,rz)
            unlink(instance.cameraNode)
        end

        instance.rotYSteeringRotSpeed = math.rad(Utils.getNoNil(getXMLFloat(xmlFile, key .. "#rotYSteeringRotSpeed"), 0))

        if instance.rotateNode == nil or instance.rotateNode == instance.cameraNode then
            instance.rotateNode = instance.cameraPositionNode
        end

        if useHeadTracking then
            local dx, dy, dz = localDirectionToLocal(instance.cameraPositionNode, getParent(instance.cameraPositionNode), 0, 0, 1)
            local tx, ty, tz = localToLocal(instance.cameraPositionNode, getParent(instance.cameraPositionNode), 0, 0, 0)

            instance.headTrackingNode = createTransformGroup("headTrackingNode")
            link(getParent(instance.cameraPositionNode), instance.headTrackingNode)
            setTranslation(instance.headTrackingNode, tx, ty, tz)

            if math.abs(dx) + math.abs(dz) > 0.0001 then
                setDirection(instance.headTrackingNode, dx, 0, dz, 0, 1, 0)
            else
                setRotation(instance.headTrackingNode, 0,0,0)
            end

            instance.headTrackingPitchOffset = math.acos(dy) - 1.57079
        end

        instance.origRotX, instance.origRotY, instance.origRotZ = getRotation(instance.rotateNode)
        instance.rotX = instance.origRotX
        instance.rotY = instance.origRotY
        instance.rotZ = instance.origRotZ
        instance.origTransX, instance.origTransY, instance.origTransZ = getTranslation(instance.cameraPositionNode)
        instance.transX = instance.origTransX
        instance.transY = instance.origTransY
        instance.transZ = instance.origTransZ

        local transLength = MathUtil.vector3Length(instance.origTransX, instance.origTransY, instance.origTransZ) + 0.00001
        instance.zoom = transLength
        instance.zoomTarget = transLength
        instance.zoomLimitedTarget = -1

        local trans1OverLength = 1.0/transLength
        instance.transDirX = trans1OverLength * instance.origTransX
        instance.transDirY = trans1OverLength * instance.origTransY
        instance.transDirZ = trans1OverLength * instance.origTransZ

        if instance.allowTranslation then
            if transLength <= 0.01 then
                g_universalPassenger:logPrint(UniversalPassenger.LOG_WARNING, "Invalid camera translation for camera '%s'. Distance needs to be bigger than 0.01", key)
            end
        end

        table.insert(instance.raycastNodes, instance.rotateNode)

        local i = 0
        while true do
            local raycastKey = key .. string.format(".raycastNode(%d)", i)
            if not hasXMLProperty(xmlFile, raycastKey) then
                break
            end

            local node = createTransformGroup("raycastNode" .. tostring(i + 1))
            link(i3dCameraNodes.outside, node)

            if node ~= nil then
                local raycastNodePosition = StringUtil.getVectorNFromString(getXMLString(xmlFile, raycastKey .. "#position"), 3)

                if raycastNodePosition ~= nil then
                    local raycastNodeRotation = Utils.getNoNil(StringUtil.getRadiansFromString(getXMLString(xmlFile, raycastKey .. "#rotation"), 3), {0, 0, 0})

                    setTranslation(node, unpack(raycastNodePosition))
                    setRotation(node, unpack(raycastNodeRotation))

                    table.insert(instance.raycastNodes, node)
                else
                    g_universalPassenger:logPrint(UniversalPassenger.LOG_WARNING, "No 'raycastNode' position given at %s", raycastKey)
                end
            end

            i = i + 1
        end

        local sx, sy, sz = getScale(instance.cameraNode)

        if sx ~= 1 or sy ~= 1 or sz ~= 1 then
            setScale(instance.cameraNode, 1, 1, 1)
        end

        instance.headTrackingPositionOffset = {0, 0, 0}
        instance.headTrackingRotationOffset = {0, 0, 0}

        return true
    end

    return false
end

function UniversalPassengerSpec:loadCharacterSpec(i3dCharacterNode, i3dCharacterNodeIndex, xmlFile, key)
    local character = nil

    if self.spec_universalPassenger.isGlobalVehicle then
        local spineRotation = Utils.getNoNil(StringUtil.getVectorNFromString(getXMLString(xmlFile, key .. "#spineRotation"), 3), {-90, 0, 90})
        local spineRot = string.format("%d %d %d", spineRotation[1], spineRotation[2], spineRotation[3])
        local customTargets = Utils.getNoNil(getXMLBool(xmlFile, key .. "#customTargets"), false)
        local childIndex = string.format("%s%.0f", i3dCharacterNodeIndex, getChildIndex(i3dCharacterNode))

        local poseIds = {} -- narrowFingers (Default), wideFingers, flatFingers

        poseIds.rightArm = Utils.getNoNil(getXMLString(xmlFile, key .. ".rightArm#poseId"), "")
        poseIds.leftArm = Utils.getNoNil(getXMLString(xmlFile, key .. ".leftArm#poseId"), "")

        local memXml = '<vehicle>\n    <enterable>        <characterNode node="' .. childIndex .. '|0" cameraMinDistance="1.5" ' ..
        'filename="$dataS2/character/player/player.xml" spineRotation="' .. spineRot ..'">/n' ..
        '        <target ikChain="rightFoot" targetNode="' .. childIndex ..'|1"/>/n' ..
        '        <target ikChain="leftFoot" targetNode="' .. childIndex ..'|2"/>/n' ..
        '        <target ikChain="rightArm" targetNode="' .. childIndex ..'|3" poseId="' .. poseIds.rightArm .. '"/>/n' ..
        '        <target ikChain="leftArm" targetNode="' .. childIndex ..'|4" poseId="' .. poseIds.leftArm .. '"/>/n</characterNode>\n    </enterable>\n</vehicle>'

        character = VehicleCharacter:new(self)

        local memXmlFile = loadXMLFileFromMemory("characterXmlFile", memXml)

        if character:load(memXmlFile, "vehicle.enterable.characterNode") then
            character.poseIds = poseIds
            character.hasCustomTargets = customTargets

            if customTargets then
                -- Current Targets
                local rightFootTarget = character.ikChainTargets["rightFoot"].targetNode
                local leftFootTarget = character.ikChainTargets["leftFoot"].targetNode
                local rightArmTarget = character.ikChainTargets["rightArm"].targetNode
                local leftArmTarget = character.ikChainTargets["leftArm"].targetNode

                -- New Tabled Targets.
                local transTargets, rotTargets = {}, {}

                transTargets["rightFoot"] = Utils.getNoNil(StringUtil.getVectorNFromString(getXMLString(xmlFile, key .. ".rightFoot#position"), 3), {getTranslation(rightFootTarget)})
                rotTargets["rightFoot"] = Utils.getNoNil(StringUtil.getRadiansFromString(getXMLString(xmlFile, key .. ".rightFoot#rotation"), 3), {getRotation(rightFootTarget)})
                transTargets["leftFoot"] = Utils.getNoNil(StringUtil.getVectorNFromString(getXMLString(xmlFile, key .. ".leftFoot#position"), 3), {getTranslation(leftFootTarget)})
                rotTargets["leftFoot"] = Utils.getNoNil(StringUtil.getRadiansFromString(getXMLString(xmlFile, key .. ".leftFoot#rotation"), 3), {getRotation(leftFootTarget)})
                transTargets["rightArm"] = Utils.getNoNil(StringUtil.getVectorNFromString(getXMLString(xmlFile, key .. ".rightArm#position"), 3), {getTranslation(rightArmTarget)})
                rotTargets["rightArm"] = Utils.getNoNil(StringUtil.getRadiansFromString(getXMLString(xmlFile, key .. ".rightArm#rotation"), 3), {getRotation(rightArmTarget)})
                transTargets["leftArm"] = Utils.getNoNil(StringUtil.getVectorNFromString(getXMLString(xmlFile, key .. ".leftArm#position"), 3), {getTranslation(leftArmTarget)})
                rotTargets["leftArm"] = Utils.getNoNil(StringUtil.getRadiansFromString(getXMLString(xmlFile, key .. ".leftArm#rotation"), 3), {getRotation(leftArmTarget)})

                -- Apply Targets
                for target, position in pairs (transTargets) do
                    setTranslation(character.ikChainTargets[target].targetNode, unpack(position))
                    setRotation(character.ikChainTargets[target].targetNode, unpack(rotTargets[target]))
                end
            end
        end

        delete(memXmlFile)
    else
        character = VehicleCharacter:new(self)

        if character:load(xmlFile, key) then
            -- Correct the 'ikChainTargets' as the ones from 'vehicle.characterNode' are loaded by default.

            character.ikChainTargets = {}
            IKUtil.loadIKChainTargets(xmlFile, key, self.components, character.ikChainTargets, self.i3dMappings)
        end
    end

    return character
end

function UniversalPassengerSpec.getLinkNodeToUse(vehicle, linkNodeStr)
    if linkNodeStr == nil then
        return vehicle.components[1].node, "0>"
    end

    -- local linkNode = I3DUtil.indexToObject(vehicle.components, linkNodeStr, vehicle.i3dMappings)
    local linkNode = UniversalPassengerSpec.getValidNodeFromIndexString(vehicle.components, linkNodeStr, vehicle.i3dMappings, nil)

    if linkNode ~= nil then
        if vehicle.i3dMappings[linkNodeStr] ~= nil then
            linkNodeStr = vehicle.i3dMappings[linkNodeStr]
        end

        return linkNode, linkNodeStr .. "|"
    else
        g_universalPassenger:logPrint(UniversalPassenger.LOG_INFO, "Link node '%s' could not be found or mod has been changed! Using 'component 1' instead, this may have undesired results. (%s)", linkNodeStr, vehicle.configFileName)

        return vehicle.components[1].node, "0>"
    end
end

function UniversalPassengerSpec.getValidNodeFromIndexString(components, index, i3dMappings, validName)
    local childNode, isValid = nil, false

    if components ~= nil and index ~= nil then
        if i3dMappings ~= nil and i3dMappings[index] ~= nil then
            index = i3dMappings[index]
        end

        local componentIndex = tonumber(index:sub(1, 1))

        if componentIndex ~= nil then
            childNode = components[componentIndex + 1].node

            if childNode ~= nil then
                local position = 3 -- Start at first character after 0>

                if #index > position then
                    while true do
                        local indexStart, indexEnd = index:find("|", position, true)

                        if indexStart == nil then
                            break
                        end

                        local currentIndex = tonumber(index:sub(position, indexStart - 1))

                        if currentIndex == nil or currentIndex >= getNumOfChildren(childNode) then
                            return nil, false
                        end

                        position = indexEnd + 1
                        childNode = getChildAt(childNode, currentIndex)
                    end

                    local currentIndex = tonumber(index:sub(position))

                    if currentIndex == nil or currentIndex >= getNumOfChildren(childNode) then
                        return nil, false
                    end

                    childNode = getChildAt(childNode, currentIndex)
                end
            end
        end
    end

    if childNode ~= nil and (validName == nil or validName == getName(childNode)) then
        isValid = true
    end

    return childNode, isValid
end

function UniversalPassengerSpec:onDelete()
    if self:getIsPassengerVehicle() then
        for id, seat in pairs (self.spec_universalPassenger.passengerSeats) do
            if seat.used then
                self:exitPassengerSeat(id, UniversalPassengerSpec.EXIT)
            end

            if seat.passengerCharacter ~= nil then
                seat.passengerCharacter:delete()
            end

            if seat.cameras ~= nil then
                for _, camera in pairs (seat.cameras) do
                    camera:delete()
                end
            end

            g_i3DManager:releaseSharedI3DFile(g_universalPassenger.baseI3D, g_universalPassenger.baseDirectory, true)
        end

        g_universalPassenger:removePassengerVehicle(self)
    end
end

function UniversalPassengerSpec:saveStatsToXMLFile(xmlFile, key)
    if self:getIsPassengerVehicle() then
        for id, seat in pairs (self.spec_universalPassenger.passengerSeats) do
            if seat.used then
                local seatKey = key .. "#passengerSeat" .. tostring(id)
                setXMLString(xmlFile, seatKey, HTMLUtil.encodeToHTML(tostring(seat.passengerName)))
            end
        end
    end

    return nil
end

function UniversalPassengerSpec:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    if self.isClient and self:getIsPassengerVehicle() then
        local spec = self.spec_universalPassenger

        if self:getIsPassenger() then
            if self:getIsControlled() then
                spec.canMoveToDriverSeat = false
            else
                spec.canMoveToDriverSeat = g_currentMission.accessHandler:canFarmAccess(g_currentMission:getFarmId(), self)
            end

            if spec.actionEventsTable.moveToDriverSeat ~= nil then
                g_inputBinding:setActionEventTextVisibility(spec.actionEventsTable.moveToDriverSeat, spec.canMoveToDriverSeat)
            end

            -- This is needed as the Drivable.lua only shows this for the person who is the vehicle.
            if self.spec_drivable ~= nil and self:getIsAIActive() then
                self:doSteeringWheelUpdate(self.spec_drivable.steeringWheel, dt, 1)
            end

            -- Alpine DLC support when train is off the track
            if spec.alpineDLCLoaded and self.trainSystem ~= nil and self.trainSystem.lastIsDrivable ~= nil and not self.trainSystem.lastIsDrivable then
                g_universalPassenger:ejectFromAlpineTrain()
            end

            self:raiseActive()
        end
    end
end

function UniversalPassengerSpec:onPostUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    if self.isClient and self:getIsPassengerVehicle() then
        local spec = self.spec_universalPassenger
        local isPassenger = self:getIsPassenger()
        local seatId

        if isPassenger then
            seatId = g_universalPassenger.currentPassengerSeatId

            if spec.passengerSeats[seatId] ~= nil and spec.activeCamera ~= nil then
                if self.firstTimeRun and spec.normalCamMode then
                    spec.activeCamera:update(dt)
                end

                if spec.activeCamera.useMirror then
                    self:setPassengerMirrorVisible(true)
                end
            end
        end

        -- Fix for Driver and other passengers not being able to see character.
        if self:getIsEntered() or isPassenger then
            for id, seat in pairs (spec.passengerSeats) do
                if seat.used and seat.passengerCharacter ~= nil then
                    local minDistance = 0

                    if isPassenger and id == seatId and seat.cameraIndex ~= 1 then
                        minDistance = 1.5 -- VehicleCharacter default.
                    end

                    seat.passengerCharacter.characterCameraMinDistance = minDistance
                    seat.passengerCharacter:updateVisibility()
                end
            end
        end
    end
end

function UniversalPassengerSpec:doSteeringWheelUpdate(steeringWheel, dt, direction)
    if steeringWheel ~= nil then
        local maxRotation = steeringWheel.outdoorRotation
        local activeCamera = self.spec_universalPassenger.activeCamera

        if activeCamera ~= nil and activeCamera.isInside then
            maxRotation = steeringWheel.indoorRotation
        end

        if self.rotatedTime == nil then
            self.rotatedTime = 0
        end

        local rotation = self.rotatedTime * maxRotation

        if steeringWheel.lastRotation ~= rotation then
            steeringWheel.lastRotation = rotation
            setRotation(steeringWheel.node, 0, rotation * direction, 0)
        end
    end
end

function UniversalPassengerSpec:setPassengerAccessState(farmAccessOnly)
    self.spec_universalPassenger.farmAccessOnly = farmAccessOnly
end

function UniversalPassengerSpec:setConnectionToSeatId(connection, seatId, isMove)
    if self.isServer then
        self.spec_universalPassenger.connectionToSeatId[connection] = seatId

        if UniversalPassenger.DEVELOPMENT_MODE == true then
            if seatId ~= nil then
                if g_currentMission.connectionsToPlayer ~= nil and g_currentMission.connectionsToPlayer[connection] ~= nil then
                    local player = g_currentMission.connectionsToPlayer[connection]

                    if isMove == true then
                        g_universalPassenger:logPrint(UniversalPassenger.LOG_DEV_DEBUG, "%s just moved to seat id %d!", player.visualInformation.playerName, seatId)
                    else
                        g_universalPassenger:logPrint(UniversalPassenger.LOG_DEV_DEBUG, "%s just entered seat id %d!", player.visualInformation.playerName, seatId)
                    end
                end
            else
                if g_currentMission.connectionsToPlayer ~= nil and g_currentMission.connectionsToPlayer[connection] ~= nil then
                    local player = g_currentMission.connectionsToPlayer[connection]
                    local seatId = self.spec_universalPassenger.connectionToSeatId[connection]

                    if seatId ~= nil then
                        g_universalPassenger:logPrint(UniversalPassenger.LOG_DEV_DEBUG, "%s just left seat id %d!", player.visualInformation.playerName, seatId)
                    end
                end
            end
        end
    end
end

function UniversalPassengerSpec:getIsPassengerVehicle()
    return self.spec_universalPassenger ~= nil and self.spec_universalPassenger.isPassengerVehicle
end

function UniversalPassengerSpec:getHasPassengers()
    return self.spec_universalPassenger.availablePassengerSeats < self.spec_universalPassenger.numPassengerSeats
end

function UniversalPassengerSpec:getIsPassenger(seatId)
    if g_currentMission.player ~= nil and g_currentMission.controlledVehicle == nil then
        if g_universalPassenger.currentPassengerVehicle == self then
            if seatId ~= nil then
                return g_universalPassenger.currentPassengerSeatId == seatId
            end

            return true
        end
    end

    return false
end

function UniversalPassengerSpec:getNextSeat(seatId)
    local spec = self.spec_universalPassenger
    local numSeats = spec.numPassengerSeats

    -- Error protection.
    if numSeats < 1 then
        return 0
    end

    if numSeats < 2 then
        -- Single seat vehicles.
        if not spec.passengerSeats[1].used then
            return 1
        end
    else
        local oldSeatId = 1
        local newSeatId = 1
        local emptySeat = false

        -- If already in seat move up one or to the start again.
        if seatId ~= nil then
            oldSeatId = seatId
            newSeatId = seatId + 1

            if newSeatId > numSeats then
                newSeatId = 1
            end
        end

        -- Use repeat so it at least checks once.
        -- Break loop if empty or original seat is found.
        repeat
            if spec.passengerSeats[newSeatId].used then
                newSeatId = newSeatId + 1

                if newSeatId > numSeats then
                    newSeatId = 1
                end
            else
                emptySeat = true
            end
        until emptySeat or newSeatId == oldSeatId

        -- If loop broke because we have empty seat then return this ID.
        if emptySeat then
            return newSeatId
        end
    end

    return 0
end

function UniversalPassengerSpec:getIsSeatAvailable(seatId)
    if self:getIsPassengerVehicle() and seatId ~= nil then
        local seat = self.spec_universalPassenger.passengerSeats[seatId]

        if seat ~= nil and not seat.used then
            return true
        end
    end

    return false
end

function UniversalPassengerSpec:setPassengerCameraIndex(seat, index)
    local spec = self.spec_universalPassenger

    if spec.activeCamera ~= nil then
        spec.activeCamera:onDeactivate()
    end

    seat.cameraIndex = index

    if seat.cameraIndex > seat.numCameras then
        seat.cameraIndex = 1
    end

    local activeCamera = seat.cameras[seat.cameraIndex]

    spec.activeCamera = activeCamera
    activeCamera:onActivate()
    g_soundManager:setIsIndoor(not activeCamera.useOutdoorSounds)

    self:setPassengerMirrorVisible(activeCamera.useMirror)

    if activeCamera.isInside then
        g_depthOfFieldManager:setManipulatedParams(nil, 0.6, nil, nil, nil)
    else
        g_depthOfFieldManager:reset()
    end
end

function UniversalPassengerSpec:getNumPassangers()
    return self.spec_universalPassenger.numPassengers or 0
end

function UniversalPassengerSpec:getAvailableSeats()
    return self.spec_universalPassenger.availablePassengerSeats
end

function UniversalPassengerSpec:setPassengerMirrorVisible(visible)
    local spec = self.spec_enterable

    if next(spec.mirrors) == nil then
        return
    end

    if visible then
        local numVisibleMirrors = 0
        local gpSpec = self.spec_universalPassenger
        for _, mirror in pairs(spec.mirrors) do
            if getIsInCameraFrustum(mirror.node, gpSpec.activeCamera.cameraNode, g_presentedScreenAspectRatio) then
                local dirX, dirY, dirZ = localToLocal(gpSpec.activeCamera.cameraNode, mirror.node, 0, 0, 0)
                dirY = dirY * g_screenAspectRatio
                local length = MathUtil.vector3Length(dirX, dirY, dirZ)
                mirror.cosAngle = -dirZ / length
            else
                mirror.cosAngle = math.huge
            end
        end

        table.sort(spec.mirrors,
                   function(mirror1, mirror2)
                        if mirror1.prio == mirror2.prio then
                            return mirror1.cosAngle > mirror2.cosAngle
                        else
                            return mirror1.prio < mirror2.prio
                        end
                   end)

        local maxNumMirrors = g_gameSettings:getValue("maxNumMirrors")

        for _, mirror in ipairs(spec.mirrors) do
            if mirror.cosAngle ~= math.huge and numVisibleMirrors < maxNumMirrors then
                setVisibility(mirror.node, true)
                numVisibleMirrors = numVisibleMirrors + 1
            else
                setVisibility(mirror.node, false)
            end
        end
    else
        for _, mirror in pairs(spec.mirrors) do
            setVisibility(mirror.node, false)
        end
    end
end

function UniversalPassengerSpec:getAllowCharacterVisibilityUpdate(superFunc)
    if g_universalPassenger ~= nil and g_universalPassenger.currentPassengerVehicle ~= nil then
        return g_universalPassenger.currentPassengerVehicle ~= self
    end

    return superFunc(self)
end

function UniversalPassengerSpec:updateNumPassangers(forceUpdate)
    local spec = self.spec_universalPassenger

    if spec ~= nil then
        local count = 0

        for _, seat in pairs (spec.passengerSeats) do
            if seat.used then
                count = count + 1
            end
        end

        spec.numPassengers = count
        spec.canMoveToDriverSeat = false
        spec.availablePassengerSeats = math.max(spec.numPassengerSeats - count, 0)
        spec.canChangeSeats = spec.availablePassengerSeats > 0

        if spec.actionEventsTable.nextPassengerSeat ~= nil and (forceUpdate or self:getIsPassenger()) then
            g_inputBinding:setActionEventActive(spec.actionEventsTable.nextPassengerSeat, spec.canChangeSeats)
            g_inputBinding:setActionEventTextVisibility(spec.actionEventsTable.nextPassengerSeat, spec.canChangeSeats)
        end
    end
end

function UniversalPassengerSpec:enterPassengerSeat(isPassenger, seatId, player)
    if g_universalPassenger:setPlayerToVehicle(player, self, isPassenger, seatId) then
        if isPassenger then
            g_universalPassenger:addPassengerActionEvents(self)

            -- Fix for driver not showing when leaving and entering vehicle again for passenger.
            local vehicleCharacter = self:getVehicleCharacter()

            if vehicleCharacter ~= nil then
                self.spec_universalPassenger.originalCharacterCameraMinDistance = vehicleCharacter.characterCameraMinDistance
                vehicleCharacter.characterCameraMinDistance = 0
            end
        end

        self:updateSeat(isPassenger, self.spec_universalPassenger.passengerSeats[seatId], true, player)
    end
end

function UniversalPassengerSpec:cyclePassengerSeats(oldSeatId, newSeatId)
    if oldSeatId == newSeatId then
        return
    end

    local spec = self.spec_universalPassenger
    local oldPlayer = spec.passengerSeats[oldSeatId].player
    local isPassenger = self:getIsPassenger(oldSeatId)

    if isPassenger then
        g_universalPassenger.currentPassengerSeatId = newSeatId
    end

    self:updateSeat(isPassenger, spec.passengerSeats[oldSeatId], false)
    self:updateSeat(isPassenger, spec.passengerSeats[newSeatId], true, oldPlayer)
end

function UniversalPassengerSpec:exitPassengerSeat(seatId, exitVehicleId, targetPosX, targetPosY, targetPosZ, isAbsolute, isRootNode)
    if seatId == nil then
        return
    end

    local spec = self.spec_universalPassenger
    local isPassenger = self:getIsPassenger(seatId)
    local seat = spec.passengerSeats[seatId]

    g_universalPassenger:setPlayerToVehicle(seat.player, nil, isPassenger, seatId)

    if isPassenger then
        if exitVehicleId ~= UniversalPassengerSpec.MOVE_VEHICLE then
            g_inputBinding:beginActionEventsModification(Player.INPUT_CONTEXT_NAME)

            g_universalPassenger:removePassengerActionEvents(self)

            if spec.activeCamera ~= nil then
                spec.activeCamera:onDeactivate()

                g_soundManager:setIsIndoor(false)
                g_depthOfFieldManager:reset()

                spec.activeCamera = nil
            end

            g_inputBinding:endActionEventsModification()

            -- Restore fix for driver not showing when leaving and entering vehicle again for passenger.
            local vehicleCharacter = self:getVehicleCharacter()

            if vehicleCharacter ~= nil then
                vehicleCharacter.characterCameraMinDistance = spec.originalCharacterCameraMinDistance or 1.5
                spec.originalCharacterCameraMinDistance = nil
            end
        end

        if exitVehicleId == UniversalPassengerSpec.EXIT then
            local player = g_currentMission.player

            if targetPosX ~= nil and targetPosY ~= nil and targetPosZ ~= nil then
                player:moveTo(targetPosX, targetPosY, targetPosZ, isAbsolute, isRootNode)
            else
                local exitPoint

                if spec.passengerSeats[seatId] ~= nil then
                    exitPoint = spec.passengerSeats[seatId].passengerExitNode
                end

                if exitPoint ~= nil then
                    local x, y, z = getWorldTranslation(exitPoint)
                    local dx, _, dz = localDirectionToWorld(exitPoint, 0, 0, -1)
                    local terrainHeight = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, x, 300, z)

                    y = math.max(terrainHeight + 0.1, y + 0.9)
                    player:moveToAbsolute(x, y, z)

                    player.rotY = MathUtil.getYRotationFromDirection(dx, dz)
                    setRotation(player.cameraNode, player.rotX, player.rotY, 0)
                else
                    player:moveToExitPoint(self)
                end
            end

            player:onEnter(true)
        end

        self:setPassengerMirrorVisible(false)
    end

    self:updateSeat(isPassenger, seat, false)
end

function UniversalPassengerSpec:updateSeat(isPassenger, seat, addPassenger, player)
    if seat ~= nil then
        if addPassenger then
            seat.player = player
            seat.used = player ~= nil

            if isPassenger then
                if g_gameSettings:getValue("isHeadTrackingEnabled") and isHeadTrackingAvailable() then
                    for camIndex, camera in pairs(seat.cameras) do
                        if camera.isInside then
                            seat.cameraIndex = camIndex

                            break
                        end
                    end
                end

                if g_gameSettings:getValue("resetCamera") then
                    seat.cameraIndex = 1
                end

                self:setPassengerCameraIndex(seat, seat.cameraIndex)
            end

            if player ~= nil and player.visualInformation ~= nil then
                seat.playerStyle = player.visualInformation
                seat.passengerName = seat.playerStyle.playerName

                local playerModel = g_playerModelManager:getPlayerModelByIndex(seat.playerStyle.selectedModelIndex)
                self:setPassengerCharacter(seat, playerModel.xmlFilename, seat.playerStyle)
            else
                seat.playerStyle = PlayerStyle:new()
                seat.passengerName = " "

                local aiPlayerModelIndex = math.random(1, g_playerModelManager:getNumOfPlayerModels())
                local playerModel = g_playerModelManager:getPlayerModelByIndex(aiPlayerModelIndex)
                self:setPassengerCharacter(seat, playerModel.xmlFilename, seat.playerStyle)
            end

            if seat.passengerEnterAnimation ~= nil and self.playAnimation ~= nil then
                self:playAnimation(seat.passengerEnterAnimation, 1, nil, true)
            end

            if seat.partChange ~= nil and seat.partChange.index ~= nil then
                setTranslation(seat.partChange.index, unpack(seat.partChange.transMax))
                setRotation(seat.partChange.index, unpack(seat.partChange.rotMax))
                setScale(seat.partChange.index, unpack(seat.partChange.scaleMax))
                setVisibility(seat.partChange.index, seat.partChange.visibilityMax)
            end

            -- Maybe need to add to 'onEnter' also.
            if self.spec_enterable.vehicleCharacter ~= nil and self:getIsActive() or self:getIsAIActive() then
                self.spec_enterable.vehicleCharacter:setCharacterVisibility(true)
            end
        else
            if isPassenger and self.spec_universalPassenger.activeCamera ~= nil then
                self.spec_universalPassenger.activeCamera:onDeactivate()
                g_soundManager:setIsIndoor(false)
                g_depthOfFieldManager:reset()
            end

            if seat.passengerCharacter ~= nil then
                seat.passengerCharacter:delete()
            end

            if seat.passengerEnterAnimation ~= nil and self.playAnimation ~= nil then
                self:playAnimation(seat.passengerEnterAnimation, -1, nil, true)
            end

            if seat.partChange ~= nil and seat.partChange.index ~= nil then
                setTranslation(seat.partChange.index, unpack(seat.partChange.transMin))
                setRotation(seat.partChange.index, unpack(seat.partChange.rotMin))
                setScale(seat.partChange.index, unpack(seat.partChange.scaleMin))
                setVisibility(seat.partChange.index, seat.partChange.visibilityMin)
            end

            seat.passengerName = ""
            seat.playerStyle = nil
            seat.used = false
            seat.player = nil
        end

        self:updateNumPassangers(isPassenger and addPassenger)
        self:raiseActive()
    end
end

function UniversalPassengerSpec:setPassengerCharacter(seat, xmlFilename, playerStyle)
    if seat.passengerCharacter ~= nil then
        seat.passengerCharacter:delete()
        seat.passengerCharacter:loadCharacter(xmlFilename, playerStyle)
        seat.passengerCharacter:updateVisibility()
        seat.passengerCharacter:updateIKChains()
    end
end

function UniversalPassengerSpec:drawUIInfo(superFunc)
    superFunc(self)

    if self:getIsPassengerVehicle() and self:getNumPassangers() > 0 and not g_flightAndNoHUDKeysEnabled then
        local step = 0.2 -- Stop long names mashing together.
        local currentSeatId = 0

        if g_universalPassenger.currentPassengerVehicle == self then
            currentSeatId = g_universalPassenger.currentPassengerSeatId or 0
        end

        for id, seat in pairs (self.spec_universalPassenger.passengerSeats) do
            if seat.used and seat.passengerName ~= nil and id ~= currentSeatId then
                local x, y, z = getWorldTranslation(seat.passengerRenderNode)
                local x1, y1, z1 = getWorldTranslation(getCamera())

                if MathUtil.vector3LengthSq(x - x1, y - y1, z - z1) <= 100 * 100 then
                    x = x + seat.passengerRenderNodeOffset[1]
                    y = y + seat.passengerRenderNodeOffset[2] + step
                    z = z + seat.passengerRenderNodeOffset[3]

                    Utils.renderTextAtWorldPosition(x,y,z, seat.passengerName, getCorrectTextSize(0.02), 0)
                end

                step = step + 0.2
            end
        end
    end
end

function UniversalPassengerSpec:playAnimation(superFunc, name, speed, animTime, noEventSend)
    if self.spec_enterable.enterAnimation == name and self:getIsAIActive() and speed < 0 then
        return -- Fix for enter animation returning to start when a driver leaves and an AI is operating.
    end

    return superFunc(self, name, speed, animTime, noEventSend)
end

function UniversalPassengerSpec:onAIEnd()
    local enterableSpec = self.spec_enterable

    if not enterableSpec.isControlled and enterableSpec.enterAnimation ~= nil and self.playAnimation ~= nil then
        self:playAnimation(enterableSpec.enterAnimation, -1, nil, true) -- Make sure the animation resets when the AI ends.
    end
end
